#uuid "A0001033"
#includecore "jsxt-string.js"
#include "hvac.js"
#include "hvac-modbus-lg.js"
#include "watchdog.js"
#include "timer.js"
#subscribe "thermo"
#subscribe "button"

var debug;
var hvac;
var conf;                           // Config file object
var config;                         // Config file handler
var licenses = 0;                   // Used licenses zone counter
var licensed = 2;                   // Licensed zone (free)
var gwStatus = {};                  // Gateway status summary (send to web interface)
var modbusConnector;
var setHttpAuthParametersTimer;
var modbusConnectorConnectionStatus = 0;
var prefix = "M-LG";
var watchDog;
var timerManager;

function main() {

    logger.info("Starting driver");

	config = new mhXMLReader();
	conf = config.open( "conf/config.xml" );
	config.onChangeEvent(reloadConfiguration);

	timerManager = new TimerManager();

	hvac = new HVAC_Driver();
	hvac.setDriver(new MODBUS_LG_Driver());	

	watchDog = new WATCHDOG();

	if (license.isValid() == mhLicense.LICENSE_VALID) {

		licensed = license.getArgument(1);
		logger.info("MH[---] license code is valid, new managed unit(s) limit is %d".sprintf(licensed));

	} else {

		logger.warning("MH[---] WARNING!!! license code is not valid, program runs with trial limits (%d units)".sprintf(licensed));
	}  

	shared.write( "licensed", JSON.stringify( { 'licensed': licensed, 'valid': license.isValid() } ) );
		
	sendModbusParametersToExecutableDriverTimer = new mhTimer();
	sendModbusParametersToExecutableDriverTimer.setInterval( 2000 );
	sendModbusParametersToExecutableDriverTimer.setAction( synchronizeModbusParameters );
	
	loadConfiguration();
	
	modbusConnector = new mhSocket();
	modbusConnector.connectTo( "127.0.0.1", 33334, mhSocket.SOCKET_PERSISTENT );
    // +OFD
    //modbusConnector.connectTo( "192.168.147.48", 33334, mhSocket.SOCKET_PERSISTENT );
    // -OFD

	modbusConnector.onConnectEvent( onModbusConnectorConnect );
	modbusConnector.onDisconnectEvent( onModbusConnectorDisconnect );
	modbusConnector.onDataEvent( onModbusConnectorData );

	setHttpAuthParametersTimer = new mhTimer();
	setHttpAuthParametersTimer.setInterval( 2000 );
	setHttpAuthParametersTimer.setAction( synchronizeAuthParameters );
	
	hvac.setDebug( true );

	//	connect event
	function onModbusConnectorConnect() {

		logger.info( "%s_DRI[---] modbus connector connected!!".sprintf( prefix ) );
		modbusConnectorConnectionStatus = 1;
		hvac.setModbusConnectorConnectionStatus( modbusConnectorConnectionStatus );
		setHttpAuthParametersTimer.start();
	}
	
	//	disconnect event
	function onModbusConnectorDisconnect() {

		logger.info( "%s_DRI[---] modbus connector disconnected!!".sprintf( prefix ) );
		modbusConnectorConnectionStatus = 0;
		hvac.setModbusConnectorConnectionStatus( modbusConnectorConnectionStatus );
	}

	//	data event
	function onModbusConnectorData( data, length ) {

		try {

			var msgObj = JSON.parse( data );
			
			var requestData = msgObj.requestData;
			var responseData = msgObj.responseData;
			
			hvac.parseResponseData( requestData, responseData );
			
		} catch (exception) {

		    logger.warning( '%s onModbusConnectorData WARNING!! exception description: %s'.sprintf( prefix, exception ) );
		}
	}
}

function synchronizeAuthParameters() {
	
	logger.fine( "%s synchronizeAuthParameters".sprintf( prefix ) );
	
	if( modbusConnectorConnectionStatus == 1 ) {
		
		// start sending tcp modbus gateway ip address and port to executable driver
		hvac.setHttpConnectorParameters();
		setHttpAuthParametersTimer.stop();
		
		logger.fine( "%s setHttpAuthParametersTimer stop".sprintf( prefix ) );
	}
}

function reloadConfiguration() {

	logger.info("MH[---] Configuration changed, restart!");
	mhOWN.exit();
}

function loadConfiguration() {

    try {

		if (parseInt(conf.preferences.bacnetserver_port.value, 10) != 0) {

			hvac.startBacnet(parseInt(conf.preferences.bacnetserver_port.value, 10));
			logger.info("MH[---] loadConfiguration F450 server configured on port <%d>".sprintf( parseInt( conf.preferences.bacnetserver_port.value, 10)));
		}

	} catch (ex) {

	    logger.error("MH[---] loadConfiguration F450 server is not configured");
	}

    try {

        if( typeof conf.preferences.ambient_reference.value != 'undefined' ) {

			if (String(conf.preferences.ambient_reference.value) == 'MH') {

				hvac.mh_as_reference = true;
				logger.info("MH[---] loadConfiguration ambient reference is <My Home>");
			}
			
			if (String(conf.preferences.ambient_reference.value) == 'HVAC') {

				hvac.mh_as_reference = false;
				logger.info("MH[---] loadConfiguration ambient reference is <HVAC System>");
			}
		}

	} catch (ex) {

		logger.error("MH[---] loadConfiguration ambient reference is not configured, <My Home> will be used");
		hvac.mh_as_reference = true;
	}
  
    try {

		if (typeof conf.preferences.heat_rangepid.value != 'undefined') {

            hvac.heat_fandelta = parseFloat( conf.preferences.heat_rangepid.value );
            logger.info("MH[---] loadConfiguration preferences heat fan delta: %.1f".sprintf(hvac.heat_fandelta));
        }

		if (typeof conf.preferences.cool_rangepid.value != 'undefined') {

            hvac.cool_fandelta = parseFloat(conf.preferences.cool_rangepid.value);
            logger.info("MH[---] loadConfiguration preferences cool fan delta: %.1f".sprintf(hvac.cool_fandelta));
        }

		if (typeof conf.preferences.heat_integratedeviation.value != 'undefined') {

            hvac.heat_integrate_delta = parseFloat(conf.preferences.heat_integratedeviation.value);
            logger.info("MH[---] loadConfiguration preferences heat integration delta: %.1f".sprintf(hvac.heat_integrate_delta));
        }

		if (typeof conf.preferences.cool_integratedeviation.value != 'undefined') {

            hvac.cool_integrate_delta = parseFloat(conf.preferences.cool_integratedeviation.value);
            logger.info("MH[---] loadConfiguration preferences cool integration delta: %.1f".sprintf(hvac.cool_integrate_delta));
        }

		if (typeof conf.preferences.heat_hysteresis != 'undefined') {

            hvac.heat_hysteresis = parseFloat( conf.preferences.heat_hysteresis.value );
            logger.info("MH[---] loadConfiguration preferences heating hysteresis value: %.1f".sprintf( hvac.heat_hysteresis ));
        }

		if (typeof conf.preferences.cool_hysteresis != 'undefined') {

            hvac.cool_hysteresis = parseFloat( conf.preferences.cool_hysteresis.value );
            logger.info("MH[---] loadConfiguration preferences cooling hysteresis value: %.1f".sprintf( hvac.cool_hysteresis ));
        }
    
		if (typeof conf.preferences.protection_mode.value != 'undefined') {

			if (String(conf.preferences.protection_mode.value).toUpperCase() == "OFF") {

		        hvac.off_thermal_protection = true;
				logger.info("MH[---] loadConfiguration preferences protection mode: 'OFF'");

			} else {

		        hvac.off_thermal_protection = false;
		        logger.info("MH[---] loadConfiguration preferences protection mode: 'Keep protection setpoint'");
			}
		}

		if( typeof (conf.preferences.heat_setpoint_reached) != 'undefined' ) {

			if (typeof conf.preferences.heat_setpoint_reached.value != 'undefined') {

		        hvac.when_reached_heat = parseInt( conf.preferences.heat_setpoint_reached.value, 10 );
		        logger.info("MH[---] loadConfiguration preferences heat setpoint reached: %s".sprintf(HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(hvac.when_reached_heat)]));
		    }
		}
		
		if( typeof conf.preferences.cool_setpoint_reached != 'undefined' ) {

			if (typeof conf.preferences.cool_setpoint_reached.value != 'undefined') {

		        hvac.when_reached_cool = parseInt( conf.preferences.cool_setpoint_reached.value, 10 );
		        logger.info("MH[---] loadConfiguration preferences cool setpoint reached: %s".sprintf(HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(hvac.when_reached_cool)]));
		    }
		}

	} catch (ex) {

		logger.error("MH[---] loadConfiguration error parsing preferences, exception description: %s".sprintf(ex));
	}  
  
    if (typeof conf.gateways.gateway != 'undefined') {

		hvac.setIPAddress(conf.gateways.gateway[0].ipaddress);
		hvac.setPort(conf.gateways.gateway[0].port);
    
		if (typeof (conf.gateways.gateway[0].type) != 'undefined') {

			hvac.hvacSetType(conf.gateways.gateway[0].type);
		}

		if (typeof (conf.gateways.gateway[0].identifier) != 'undefined') {

			hvac.hvacSetIdentifier(conf.gateways.gateway[0].identifier);
		}

	} else {

		logger.error("MH[---] loadConfiguration gateway configuration missing, halt!");
		return;
	}
  
    if (typeof conf.zones.zone != 'undefined') {

        for (x = 0; x < conf.zones.zone.length; x++) {

            licenses++;

			if (licenses > licensed) {

				logger.error("MH[---] loadConfiguration licensed zone limit (%d) reached, skipping %d^ zone".sprintf(licensed, licenses));

			} else {

				logger.info("MH[---] loadConfiguration zone %d^ licensed (%d limit)".sprintf(licenses,licensed));
				var item = conf.zones.zone[x];

				var hvacSlaveAddressArray = Array();
				if (typeof item.hvacslaves != 'undefined') {

				    hvacSlaveAddressArray.push(parseInt(item.hvacslaves.hvacaddress_1,10));
				    hvacSlaveAddressArray.push(parseInt(item.hvacslaves.hvacaddress_2,10));
				    hvacSlaveAddressArray.push(parseInt(item.hvacslaves.hvacaddress_3,10));
				    hvacSlaveAddressArray.push(parseInt(item.hvacslaves.hvacaddress_4,10));
				    hvacSlaveAddressArray.push(parseInt(item.hvacslaves.hvacaddress_5,10));
				}

				var id = hvac.addZone(item.ownserver, item.probeaddress, item.bacnetarea, item.bacnetaddress, item.hvacid, item.hvacaddress, hvacSlaveAddressArray, (String(item.autofanspeed).toUpperCase() == "YES") ? true:false);
				
				hvac.editZone_Fanprobe(id,(String(item.probefanmode).toUpperCase() == "YES") ? true:false);
        
				if (item.startupfanspeed) {

					logger.info("MH[gw:%02d,z:%02d] set default fan speed: %d".sprintf(item.ownserver, item.probeaddress, item.startupfanspeed));
					hvac.setFanSpeed(id,item.startupfanspeed);
				}

				if (item.hvactype) {

					logger.info("MH[gw:%02d,z:%02d] set unit type: %d".sprintf(item.ownserver, item.probeaddress, item.hvactype));
					hvac.editZone_HVAC_Type(id,item.hvactype);
				}

				if (typeof item.name != 'undefined') {

					hvac.editZone_Name(id,String(item.name));
				}

				if (typeof item.heating != 'undefined') {

					hvac.editZone_Heating_Details(id, String(item.heating.type), String(item.heating.mode), parseInt(item.heating.defaultairdir,10));
				}
        
				if (typeof item.cooling != 'undefined') {

					hvac.editZone_Cooling_Details(id, String(item.cooling.type), String(item.cooling.mode), parseInt(item.cooling.defaultairdir,10));
				}

				if (typeof item.adjust != 'undefined') {

					/* Backward compatibility, uses defaults if this value does not exist */
					hvac.editZone_AdjustMode( id, String( item.adjust ) );
				}

				if (typeof item.locking != 'undefined') {

				    hvac.editZone_Lock( id,
                                        String(item.locking.cenaddress),
                                        parseInt(item.locking.cenplusaddress, 10),
					                    parseInt(item.locking.buttonlock, 10),
                                        parseInt(item.locking.buttonunlock, 10),
                                        parseInt(item.locking.buttontemporary, 10));
				}
			}
        }

	} else {

		logger.error("MH[---] loadConfiguration No zone defined, halt!");
		return;
	}
  
    if(typeof conf.ownservers.ownserver!='undefined') {

        for(x = 0; x < conf.ownservers.ownserver.length; x++) {

			var own = conf.ownservers.ownserver[x];
      
			logger.log(format("MH[gw:%02d] Connecting to My Home Gateway %s:%s with password %s",own.id,own.ipaddress,own.port,own.pwd));
			if (mhOWN.connectTo(own.id, own.ipaddress, own.port, own.pwd) == -1) {

				logger.error("MH[---] loadConfiguration Unable to allocate connection");

			}

			gwStatus[own.id] = {};
			gwStatus[own.id]['id'] = parseInt(own.id,10);
			gwStatus[own.id]['name'] = String(own.name);
			gwStatus[own.id]['address'] = format("%s (%s:%d)",own.name,own.ipaddress,own.port);
			gwStatus[own.id]['connected'] = false;

			if (typeof (own.connections) != 'undefined') {

				gwStatus[own.id]['connections'] = parseInt(own.connections, 10);

			} else {

				gwStatus[own.id]['connections'] = 1;
			}

			shared.write('ownservers', JSON.stringify(gwStatus));
        }

	} else {

		logger.info("MH[---] loadConfiguration No external openserver defined");
	}

	try {

		// watch dog parameters
		if (typeof conf.watchdogs != 'undefined') {

			logger.info("MH[---] loadConfiguration conf.watchdogs recognized");

			if (typeof conf.watchdogs.watchdog != 'undefined') {

				logger.info("MH[---] loadConfiguration conf.watchdogs.watchdog recognized");

				if (typeof (conf.watchdogs.watchdog[0].timeout) != 'undefined') {

					watchDog.setTimeout(parseInt(conf.watchdogs.watchdog[0].timeout, 10));
				}

				if (typeof (conf.watchdogs.watchdog[0].alarm) != 'undefined') {

					watchDog.setAlarm(parseInt(conf.watchdogs.watchdog[0].alarm, 10));
				}

				if (typeof (conf.watchdogs.watchdog[0].reboot) != 'undefined') {

					watchDog.setReboot(parseInt(conf.watchdogs.watchdog[0].reboot, 10));
				}

				if (typeof (conf.watchdogs.watchdog[0].ignore) != 'undefined') {

					watchDog.setIgnore(parseInt(conf.watchdogs.watchdog[0].ignore, 10));
				}

				if (typeof (conf.watchdogs.watchdog[0].buffer) != 'undefined') {

					watchDog.setBuffer(parseInt(conf.watchdogs.watchdog[0].buffer, 10));
				}

				if (typeof (conf.watchdogs.watchdog[0].modbus) != 'undefined') {

					//watchDog.MODBUS = parseInt(conf.watchdogs.watchdog[0].modbus, 10);
					//logger.info("MH[---] loadConfiguration watchdog buffer config value: %d".sprintf(watchDog.MODBUS));
				}

			} else {

				watchDog.setTimeout(0);
				logger.warning("MH[---] loadConfiguration watch dog configuration missing");
			}
		}

	} catch (ex) {

		logger.error("MH[---] loadConfiguration watch dog configuration exception: %s".sprintf(ex));
	}

	watchDog.endOfConfig(); // start watch dog operations
  
	hvac.endOfConfig(); /* Call fetch operations */
}

function synchronizeModbusParameters() {
	
	logger.fine( "%s synchronizeModbusParameters".sprintf( prefix ) );
	
	if( modbusConnectorConnectionStatus == 1 ) {
		
		//httpConnectorAuthenticationStatus = dataManager.setHttpConnectorParameters( devicesParameters );
		sendModbusParametersToExecutableDriverTimer.stop();
		
		logger.fine( "%s synchronizeModbusParameters sendModbusParametersToExecutableDriverTimer stop".sprintf( prefix ) );
	}
}


function mhOWN_onConnect(gatewayId) {

	logger.info(format("MH[gw:%02d] My Home Gateway connected", gatewayId));
	
	if (gatewayId > 0) {

		gwStatus[gatewayId]['connected'] = true;
	}
	
	if( typeof conf.zones.zone != 'undefined' ) {

	    for( x = 0; x < conf.zones.zone.length; x++ ) {

			var item = conf.zones.zone[x];
			if( item.ownserver == gatewayId ) {

			    //+FEAT. Er2 Management
			    hvac.setForcedLedUpdate(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId,item.probeaddress),true);
			    //-FEAT. Er2 Management

				mhOWN.sendFrame( gatewayId, format( "*#4*%d##", item.probeaddress ) );
				
				if (String(item.probefanmode).toUpperCase() == "YES") {

					mhOWN.sendFrame( gatewayId, format( "*#4*%d*11##", item.probeaddress ) );
				}
		
				// Fix frame 17
				//logger.info("MH[gw:%02d,z:%02d] send force updateled".sprintf(gatewayId,item.probeaddress));
				//hvac.updateLed(gatewayId,true);
			}
		}
	}

	shared.write('ownservers',JSON.stringify(gwStatus));
}

function mhOWN_onDisconnect(gatewayId) {

    logger.info(format("MH[gw:%02d] My Home Gateway disconnected", gatewayId));
    
    if (gatewayId > 0) {

        gwStatus[gatewayId]['connected'] = false;
        shared.write('ownservers',JSON.stringify(gwStatus));
    }

    //+FEAT. Er2 Management
    if(typeof conf.zones.zone!='undefined') {

        for(x = 0; x < conf.zones.zone.length; x++) {

            var item = conf.zones.zone[x];
			if (item.ownserver == gatewayId) {

                hvac.setForcedLedUpdate(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, item.probeaddress), false);
            }
        }
    }
    //-FEAT. Er2 Management
}

//function mhOWN_onFrame(gatewayId, frame) {
    
//    logger.debug("MH[gw:%02d] mhOWN_onFrame frame: %s".sprintf(gatewayId,frame));
    
//    /* *25*21#<button>*2<where>## */
//	if (matches = frame.match(/\*25\*21#(\d+)\*2(\d+)##/)) {

//        hvac.cenplusEvent(gatewayId, parseInt(matches[2], 10), parseInt(matches[1], 10));
//    }
//}

function mhOWN_onFrame(gatewayId, frame) {

	var matches;

	//if (matches = frame.match(/\*25\*(\d+)#(\d+)\*2(\d+)##/)) {

	//	// *25*ACTION#PUSHBUTTON*VIRTUAL ADDRESS##	ACTION [ 21, 22, 23, 24 ], PUSHBUTTON [ 0 - 31 ],  VIRTUAL ADDRESS [ 2 ][ 0 - 2047 ]

	//	// Cen Plus Action
	//	// 21	Short pressure ( < 0.5 seconds )
	//	// 22	Start of extended pressure ( >= 0.5 seconds )
	//	// 23	Extended pressure
	//	// 24	Released after an extended pressure
	//	var cenPlusAction = parseInt(matches[1], 10);

	//	// value in the range [0 - 31]
	//	var virtualPushButton = parseInt(matches[2], 10);

	//	// Virtual Address has a value done with the union of:	[ 2 ] [ Object ]
	//	// Object has a value in the range [0 - 2047]
	//	var virtualAddress = parseInt(matches[3], 10);

	//	logger.info("MH[gw:%02d] recognized cen plus frame: %s, action: %d, virtual push button: %d, virtual address: %d".sprintf(gatewayId, frame, cenPlusAction, virtualPushButton, virtualAddress));

	//	if (cenPlusAction == 21) {

	//		hvac.cenplusEvent(gatewayId, virtualAddress, virtualPushButton, cenPlusAction);
	//	}

	if (matches = frame.match(/\*25\*21#(\d+)\*2(\d+)##/)) {

		// *25*21#<PUSHBUTTON>*<VIRTUAL ADDRESS>##		PUSHBUTTON [ 0 - 31 ],  VIRTUAL ADDRESS [ 2 ][ 0 - 2047 ]

		// value in the range [0 - 31]
		var virtualPushButton = parseInt(matches[1], 10);

		// Virtual Address has a value done with the union of:	[ 2 ] [ Object ]
		// Object has a value in the range [0 - 2047]
		var virtualAddress = parseInt(matches[2], 10);

		logger.info("MH[gw:%02d] recognized cen plus frame: %s, virtual push button: %d, virtual address: %d".sprintf(gatewayId, frame, virtualPushButton, virtualAddress));

		hvac.cenplusEvent(gatewayId, virtualAddress, virtualPushButton);

	} else if (matches = frame.match(/^\*4\*(\d+)\*(\d+)##/)) {

		var what = parseInt(matches[1], 10);

		var myHomeZoneAddress = parseInt(matches[2], 10);
		var mode = -1;
		var submode = -1;
		var season = 10;
		var discard = false;

		switch (what) {

			case 402:
				mode = 1;
				submode = 14;
				break;

			case 407:
				mode = 1;
				submode = 13;
				break;

			case 408:
				mode = 1;
				submode = 12;
				break;

			case 410:
				mode = 1;
				submode = 10;
				break;

			case 502:
				mode = 2;
				submode = 14;
				break;

			case 507:
				mode = 2;
				submode = 13;
				break;

			case 508:
				mode = 2;
				submode = 12;
				break;

			case 510:
				mode = 2;
				submode = 10;
				break;

			default:
				// not a change over frame! Discard it
				discard = true;
		}

		if (discard === false) {

			logger.info("MH[gw:%02d] mhOWN_onFrame recognized change over frame: %s".sprintf(gatewayId, frame));
			mhOWN_onThermoZone_OperationMode(gatewayId, myHomeZoneAddress, mode, submode, season);

		} else {

			logger.debug("MH[gw:%02d] mhOWN_onFrame frame: %s".sprintf(gatewayId, frame));
		}

	} else if (matches = frame.match(/^\*#4\*(\d+)#(\d+)\*#17\*(\d+)\*(\d+)\*(\d+)##/)) {

		// *#4*dove*#17*V*Mod*T##

		// dove = Z#N = [0-99]#[1-9]
		// V = Velocit da impostare sul fan-coil
		// Mod = Modalit di funzionamento
		// T = Temperatura di setpoint, espressa in passi di 0,5C
		logger.info("MH[gw:%02d] mhOWN_onFrame recognized gateway status setting frame: %s".sprintf(gatewayId, frame));

		var Z = parseInt(matches[1], 10);
		var N = parseInt(matches[2], 10);
		//var V = parseInt(matches[3], 10);

		var Mod = parseInt(matches[4], 10);
		//var T = parseInt(matches[5], 10);
		var T = 30;

		var zonesArrayIndex = hvac.getZoneByAddress(gatewayId, Z);

		if (zonesArrayIndex > -1) {

			var V = hvac.evaluateZoneFanSpeedForProbe(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, Z));

			// gateway state setting: *#4*dove*17*V*Mod*T##

			var frame = "*#4*%d#%d*17*%d*%d*%04d##".sprintf(Z, N, V, Mod, T);

			logger.info("MH[gw:%02d] mhOWN_onFrame sending gateway status frame: %s".sprintf(gatewayId, frame));

			mhOWN.sendFrame(gatewayId, frame);
		}

	} else {

		logger.debug("MH[gw:%02d] mhOWN_onFrame frame: %s".sprintf(gatewayId, frame));
	}
}


function mhOWN_onCEN(gatewayId, bus, a, pl, button, type) {

	if (type == mhOWN.OWN_CEN_PRESS) {

        hvac.cenEvent(gatewayId,bus,a,pl,button);
    }
}

function mhOWN_setDebug(status) {

	logger.info( "Debug is set to %s".sprintf( ( status ) ? "on" : "off" ) );
	debug = status;
	hvac.setDebug( debug );
}

function mhOWN_onThermoZone_Ambient(gatewayId, myHomeZoneAddress, temperature) {

    logger.info("MH[gw:%02d,z:%02d] received temperature: %.1f".sprintf(gatewayId, myHomeZoneAddress, temperature));
    hvac.setAmbient(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, myHomeZoneAddress), temperature);
}

function mhOWN_onThermoZone_Setpoint(gatewayId, myHomeZoneAddress, temperature) {

    logger.info("MH[gw:%02d,z:%02d] received setpoint: %.1f".sprintf(gatewayId, myHomeZoneAddress, temperature));
    hvac.setSetpoint(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, myHomeZoneAddress), temperature);
}

function mhOWN_onThermoZone_FanSpeed(gatewayId, myHomeZoneAddress, speed) {

    logger.info("MH[gw:%02d,z:%02d] received fan speed: %d".sprintf(gatewayId, myHomeZoneAddress, speed));
    hvac.setFanSpeed(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, myHomeZoneAddress), speed);
}

function mhOWN_onThermoZone_OperationMode(gatewayId, myHomeZoneAddress, mode, submode, season) {

    var modeDescription;
    var subModeDescription;

	if (mode > -1) {

		modeDescription = HVAC_Driver.modesDescription[mode];

	} else {

        modeDescription = "Unknown";
    }

	if (submode > -1) {

		subModeDescription = HVAC_Driver.modesDescription[submode];

	} else {

        subModeDescription = "Unknown";
    }

    logger.info("MH[gw:%02d,z:%02d] received operation mode: %s (%d), submode: %s (%d), season: %d".sprintf(gatewayId, myHomeZoneAddress, modeDescription, mode, subModeDescription, submode, season));
    hvac.setMode(hvac.getZoneArrayIndexByMyHomeZoneAddress(gatewayId, myHomeZoneAddress), mode, submode);
}

function mhOWN_onThermoZone_GatewayUpdateRequest(gatewayId, myHomeZoneAddress) {

    //logger.info("MH[gw:%02d,z:%02d] received gateway update request".sprintf(gatewayId, myHomeZoneAddress));

    //// check if it need a status update
    //var zoneArrayIndex = hvac.getZoneArrayIndexByMyHomeZoneAddress( gatewayId, myHomeZoneAddress );

    //if( zoneArrayIndex > -1 ) {
    //    var forcedLedUpdateStatus = hvac.getForcedLedUpdate( zoneArrayIndex );
    //    if( debug ) logger.debug( "MH[gw:%02d,z:%02d] mhOWN_onThermoZone_GatewayUpdateRequest -> forcedLedUpdateStatus: %s".sprintf( gatewayId, myHomeZoneAddress, forcedLedUpdateStatus ? "TRUE" : "FALSE" ) );
    //    hvac.ledUpdate( zoneArrayIndex, forcedLedUpdateStatus );
    //} else {
    //    logger.info("MH[gw:%02d,z:%02d] mhOWN_onThermoZone_GatewayUpdateRequest -> ALERT!! EXISTING PROBE NOT FOUND IN XML CONFIG FILE!".sprintf(gatewayId,myHomeZoneAddress));
    //}
}
