var MODBUS_LG_Driver = (function () {

    // constructor
    var drv = function () {

        var that = this;                  // Useful for off-scope calls

        var WRITE_SINGLE_COIL = 1;
        var WRITE_MULTIPLE_COILS = 4;
        var WRITE_SINGLE_REGISTER = 2;
        var WRITE_MULTIPLE_REGISTERS = 8;

        // Variables
        var baseaddress = 0;
        var driverManagerVersionFlag;
        this.zones = [];
        this.debug = false;
        this.conn;
        //this.modbusConnector;
        this.modbusConnectorConnectionStatus = 0;
        this.synchronizationArray = [];
        this.station = 10;                 	/* Used to identify station on modbus or id on multi-card systems */
        this.ipaddress = "127.0.0.1";     	/* Address of the IP interface */
        this.port = 502;                  	/* Port of the IP interface */
        this.name = "LG Modbus Driver";
        this.prefix = "M-LG";
        this.dirty_max = 3;
        this.onAmbientCallback = null;
        this.connected = false;

        this.incommit = false;
        this.interfacetype = -1;
        this.dirtylist = [];

        this.slow = 5000;                 	/* Needed interval between writes in ms */
        // originale -> this.slowread = 500;                 	/* Needed interval between reads in ms */
        this.slowread = 10000;            	/* Needed interval between reads in ms */
        this.lastwrite = 0;               	/* Last write operation timestamp */
        this.lastread = 0;               	/* Last write operation timestamp */

        /* Strings for logging purpose */
        this.s_mh_fanspeeds = []; this.s_mh_fanspeeds[0] = "INVALID"; this.s_mh_fanspeeds[1] = "LOW"; this.s_mh_fanspeeds[2] = "MIDDLE"; this.s_mh_fanspeeds[3] = "HIGH"; this.s_mh_fanspeeds[4] = "AUTO";
        this.s_mh_orientations = []; this.s_mh_orientations[0] = "INVALID";
        this.s_hvacmodes = ['COOL', 'DRY', 'FAN', 'AUTO', 'HEAT'];

        /* Map between MH ( the key ) and HVAC ( the value ) air direction modes */
        /* MH air directions 0=Auto 1=forward 2=backward 3=random 4=slow cycling 5=cycling 6=fast cycling 7=stop */
        this._orientations = [];
        this._orientations[0] = 1; this._orientations[1] = 2; this._orientations[2] = 6; this._orientations[3] = 1;
        this._orientations[4] = 1; this._orientations[5] = 1; this._orientations[6] = 1; this._orientations[7] = 7;

        // ------------------------------------------------------------------------
        // Interface
        // ------------------------------------------------------------------------

        // try {
        // logger.finer( 'driver A0001033' );
        // driverManagerVersionFlag = 1;
        // } catch( exception ) {
        // logger.info( "MH[---] driver manager doesn't support advanced logging" );
        // driverManagerVersionFlag = 0;
        // }

        //logger.info( "%s_MH[---] driverManagerVersionFlag: %d".sprintf( this.prefix, driverManagerVersionFlag ) );		

        // // cannot use executable driver to process modbus communications
        // this.conn = new mhModbusClient();
        // this.conn.onWriteSuccessEvent( function( a, b ) { that.modbusSucess( a, b ); } );
        // this.conn.onWriteErrorEvent( function( a, b, c ) { that.modbusError( a, b, c ); } );

        // this.conn.setWriteMode( WRITE_SINGLE_REGISTER );
        // this.conn.setWriteMode( WRITE_SINGLE_COIL );

        this.addToDirty = function (id) {
            if (id == -1) {
                this.dirtylist = [];
                return;
            }
            if (this.dirtylist.indexOf(id) == -1) this.dirtylist.push(id);
        }

        this.modbusConnectorSendData = function (_data) {
            //this.modbusConnector.send( JSON.stringify( _data ) );
            var json = JSON.stringify(_data);

            logger.info("%s_DRI[---] json: %s".sprintf(this.prefix, json));

            modbusConnector.send(json);
        }

        this.setHttpConnectorParameters = function () {

            logger.finest("%s +setHttpConnectorParameters".sprintf(this.prefix));

            var ipAddress = String(this.getIPAddress());
            var port = parseInt(this.getPort(), 10);

            // send to executable driver tcp modbus gateway ip address and port
            logger.info("%s_DRI[---] ipaddress: %s, port: %d".sprintf(this.prefix, ipAddress, port));

            //send ipAddress and port to connect to TCP Modbus gateway
            var modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "connect", cmdDevice: this.station, cmdString: "" }, params: { ipaddress: ipAddress, port: port, write_single_coil: 1, write_single_register: 1, time_ms: Date.now() } };

            this.modbusConnectorSendData(modbusRequestObject);

            logger.finest("%s -setHttpConnectorParameters".sprintf(this.prefix));
        }

        this.asynchronousCommit = function (id) {

            logger.info("%s_DRI[---] asynchronousCommit requested for <%d> ( incommit: %d )".sprintf(this.prefix, id, this.incommit));

            this.periodicRead.start(); // Restart periodicRead if the commitTimer start me

            if (this.modbusConnectorConnectionStatus == 0) {

                logger.info("%s_DRI[---] asynchronousCommit modbus connector not connected, skipping commit".sprintf(this.prefix));
                return;
            }

            // DMLGM-1
            // Avoid overlapping commit but place a hold-time limit
            // If a commit restart the loop, the dirtylist is not empty ( because is cleaned at the end ) and the commit restart from the first zone even if it's ( now ) clean. 
            // TODO: We have two other options:
            // - remove the single zone from dirtylist after modbus write ( good, it's not too bad )
            // - remove the zone if after fetchAll loop it's becomes "ok" 

            if (this.incommit > 0) {

                this.incommit = this.incommit + 1;
                if (this.incommit <= 5) {

                    logger.info("%s_DRI[---] asynchronousCommit commit requested for <%d> ( overlap protection in action )".sprintf(this.prefix, id));
                    return;
                }
            }

            // Commit all ( id=-1 ), only dirtylist ( -2 ) or a single zone
            logger.info("%s_DRI[---] asynchronousCommit commit requested for <%d> ( incommit: %d )".sprintf(this.prefix, id, this.incommit));
            this.incommit = 1;

            for (var i in this.zones) {

                if (id == -1 || this.zones[i].id == id || (id == -2 && this.dirtylist.indexOf(this.zones[i].id) >= 0)) {

                    // Send this zone
                    var zone = this.zones[i];

                    logger.debug("%s_DRI[u:%02d] asynchronousCommit zone.mh_setpoint: %.1f, zone.mh_mode: %d".sprintf(this.prefix, zone.address, zone.mh_setpoint, zone.mh_mode));

                    if (zone.mh_setpoint > -1 && zone.mh_mode > -1) {

                        if (zone.mh_power) {

                            switch (zone.mh_mode) {

                                // Standard modes
                                case mhOWN.OWN_THERMO_HEATING:	//	1
                                    // Ceil setpoint on heating
                                    // originale -> zone.mh_setpoint = Math.ceil( zone.mh_setpoint );
                                    zone.sent_mode = 5;
                                    break;

                                case mhOWN.OWN_THERMO_COOLING:	//	2
                                    // Floor setpoint on cooling
                                    // originale -> zone.mh_setpoint = Math.floor( zone.mh_setpoint );
                                    zone.sent_mode = 1;
                                    break;

                                case HVAC_Driver.OWN_THERMO_DRY:  //	101
                                    zone.sent_mode = 2;
                                    break;

                                case HVAC_Driver.OWN_THERMO_FAN:	//	102
                                    zone.sent_mode = 3;
                                    break;

                                case HVAC_Driver.OWN_THERMO_AUTO:	//	103
                                    zone.sent_mode = 4;
                                    break;
                            }

                        } else {

                            zone.sent_mode = 4; // Set to auto mode
                        }

                        var coil = 16 * (zone.address) + baseaddress;
                        var hold = 16 * (zone.address) + baseaddress;
                        //outbuffer.push( zone.sent_mode );

                        logger.info("%s_DRI[u:%02d] asynchronousCommit my home fan speed: %d".sprintf(this.prefix, zone.address, zone.mh_fanspeed));

                        switch (zone.mh_fanspeed) {

                            case 1:
                                zone.sent_fanspeed = 1
                                break;

                            case 2:
                                zone.sent_fanspeed = 2
                                break;

                            case 3:
                                zone.sent_fanspeed = 3
                                break;

                            case 0:
                            default:
                                zone.sent_fanspeed = 4
                                break;
                        }

                        // Orientation, -1 is "none" ( 0-8 for this driver, this._orientations maps MH 0-7 modes )
                        if (zone.mh_orientation == -1) {

                            //outbuffer.push( this._orientations[zone.mh_orientation] );
                            //sent_orientation = 0

                        } else if (zone.mh_orientation == 0) {

                            //sent_orientation = 0
                        }

                        while (this.slow > 0 && (Date.now() - this.lastwrite) < this.slow) {

                            sleep(500); // Wait 500ms
                        }

                        // Send a single write request because Hitachi systems 
                        // don't like multiple write request in a short time ( it writes all requests
                        // directly on the HVAC system bus without buffer. Other systems like Intesisbox
                        // uses a buffer and write on HVAC bus with a separated task

                        this.lastwrite = Date.now();

                        //this.conn.writeCoil( this.station, coil, zone.mh_power ? 1 : 0 );
                        var myHomePower = zone.mh_power ? 1 : 0;
                        var modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "coil" }, params: { zone: zone.address, type: 20, address: coil, value: myHomePower, time_ms: Date.now() } };
                        logger.debug("%s_DRI[u:%02d] asynchronousCommit sending modbus coil write request: %s".sprintf(this.prefix, zone.address, JSON.stringify(modbusRequestObject)));

                        this.modbusConnectorSendData(modbusRequestObject);

                        logger.info("%s_DRI[u:%02d] asynchronousCommit sent power value: %s (%d), modbus RUN/STOP coil address: %d".sprintf(this.prefix, zone.address, myHomePower ? "on" : "off", myHomePower, coil));
                        sleep(500);

                        var setPointTowrite = zone.mh_setpoint;

                        // check if my home set point exceeds hvac unit's temperature upper limit or temperature lower limit
                        if (zone.hv_temperature_upper_limit > 0 && zone.mh_setpoint > zone.hv_temperature_upper_limit) {

                            // new hvac unit's set point will be it's temperature upper limit
                            setPointTowrite = zone.hv_temperature_upper_limit;
                            logger.info("%s_DRI[u:%02d] asynchronousCommit my home set point: %d exceeds hvac unit's temperature upper limit: %d, so new hvac set point will be it's temperature upper limit".sprintf(this.prefix, zone.address, zone.mh_setpoint, setPointTowrite));

                        } else if (zone.hv_temperature_lower_limit > 0 && zone.mh_setpoint < zone.hv_temperature_lower_limit) {

                            // new hvac unit's set point will be it's temperature lower limit
                            setPointTowrite = zone.hv_temperature_lower_limit;
                            logger.info("%s_DRI[u:%02d] asynchronousCommit my home set point: %d exceeds hvac unit's temperature lower limit: %d, so new hvac set point will be it's temperature lower limit".sprintf(this.prefix, zone.address, zone.mh_setpoint, setPointTowrite));

                        } else {

                            //nothing to do here
                        }

                        //this.conn.writeRegister( this.station, hold + 2, setPointTowrite );
                        modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.address, type: 22, address: hold + 2, value: setPointTowrite, time_ms: Date.now() } };
                        logger.debug("%s_DRI[u:%02d] asynchronousCommit sending SET TEMP modbus register write request: %s".sprintf(this.prefix, zone.address, JSON.stringify(modbusRequestObject)));

                        this.modbusConnectorSendData(modbusRequestObject);

                        logger.info("%s_DRI[u:%02d] asynchronousCommit sent setpoint value: %d, modbus register address: %d".sprintf(this.prefix, zone.address, setPointTowrite, hold + 2));
                        sleep(500);

                        //this.conn.writeRegister( this.station, hold, zone.sent_mode );
                        modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.address, type: 22, address: hold, value: zone.sent_mode, time_ms: Date.now() } };
                        logger.debug("%s_DRI[u:%02d] asynchronousCommit sending OPERATION MODE modbus register write request: %s".sprintf(this.prefix, zone.address, JSON.stringify(modbusRequestObject)));

                        this.modbusConnectorSendData(modbusRequestObject);

                        logger.info("%s_DRI[u:%02d] asynchronousCommit sent mode: %s ( %d ) modbus register address: %d".sprintf(this.prefix, zone.address, HVAC_Driver.lg_modes[zone.sent_mode], zone.sent_mode, hold));
                        sleep(500);

                        //this.conn.writeRegister( this.station, hold + 1, zone.sent_fanspeed );
                        modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.address, type: 22, address: hold + 1, value: zone.sent_fanspeed, time_ms: Date.now() } };
                        logger.debug("%s_DRI[u:%02d] asynchronousCommit sending FAN SPEED modbus register write request: %s".sprintf(this.prefix, zone.address, JSON.stringify(modbusRequestObject)));

                        this.modbusConnectorSendData(modbusRequestObject);

                        logger.info("%s_DRI[u:%02d] asynchronousCommit sent fan speed: %d, modbus register address: %d".sprintf(this.prefix, zone.address, zone.sent_fanspeed, hold + 1));
                        sleep(500);

                        zone.dirty_counter = zone.dirty_counter + 1;
                        var sent_orientation = this._orientations[zone.mh_orientation];

                        logger.info("%s_DRI[u:%02d] master commit done <ambient: %.1f, setpoint: %.1f, power: %d, mode: %s( %d ), fan: %s, orientation: %s, tryes: %d, address: %d>".sprintf(this.prefix, zone.address, zone.mh_ambient, zone.mh_setpoint, zone.mh_power, HVAC_Driver.lg_modes[zone.sent_mode], zone.sent_mode, this.s_mh_fanspeeds[zone.sent_fanspeed], this.s_mh_orientations[sent_orientation], zone.dirty_counter, zone.address));

                        // +FEAT. HVAC Master/Slaves
                        // write registers for slave addresses
                        for (var zz = 0; zz < zone.slaves.length; zz++) {

                            if (zone.slaves[zz] > -1) {

                                coil = 16 * (zone.slaves[zz]) + baseaddress;
                                hold = 16 * (zone.slaves[zz]) + baseaddress;

                                sleep(50); /* Wait 50ms */

                                modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "coil" }, params: { zone: zone.slaves[zz], type: 20, address: coil, value: myHomePower, time_ms: Date.now() } };
                                logger.debug("%s_DRI[u:%02d] asynchronousCommit slave: %d, sending modbus coil write request: %s".sprintf(this.prefix, zone.address, zone.slaves[zz], JSON.stringify(modbusRequestObject)));

                                this.modbusConnectorSendData(modbusRequestObject);

                                logger.info("%s_DRI[u:%02d] asynchronousCommit slave: %d sent power value: %s (%d), modbus RUN/STOP coil address: %d".sprintf(this.prefix, zone.address, zone.slaves[zz], myHomePower ? "on" : "off", myHomePower, coil));
                                sleep(500);

                                modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.slaves[zz], type: 22, address: hold + 2, value: setPointTowrite, time_ms: Date.now() } };
                                logger.debug("%s_DRI[u:%02d] asynchronousCommit slave: %d, sending SET TEMP modbus register write request: %s".sprintf(this.prefix, zone.address, zone.slaves[zz], JSON.stringify(modbusRequestObject)));

                                this.modbusConnectorSendData(modbusRequestObject);

                                logger.info("%s_DRI[u:%02d] asynchronousCommit slave: %d, sent setpoint value: %d, modbus register address: %d".sprintf(this.prefix, zone.address, zone.slaves[zz], setPointTowrite, hold + 2));
                                sleep(500);

                                modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.slaves[zz], type: 22, address: hold, value: zone.sent_mode, time_ms: Date.now() } };
                                logger.debug("%s_DRI[u:%02d] asynchronousCommit slave: %d, sending OPERATION MODE modbus register write request: %s".sprintf(this.prefix, zone.address, zone.slaves[zz], JSON.stringify(modbusRequestObject)));

                                this.modbusConnectorSendData(modbusRequestObject);

                                logger.info("%s_DRI[u:%02d] asynchronousCommit slave: %d, sent mode: %s ( %d ) modbus register address: %d".sprintf(this.prefix, zone.address, zone.slaves[zz], HVAC_Driver.lg_modes[zone.sent_mode], zone.sent_mode, hold));
                                sleep(500);

                                modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "write", cmdDevice: this.station, cmdString: "register" }, params: { zone: zone.slaves[zz], type: 22, address: hold + 1, value: zone.sent_fanspeed, time_ms: Date.now() } };
                                logger.debug("%s_DRI[u:%02d] asynchronousCommit slave: %d, sending FAN SPEED modbus register write request: %s".sprintf(this.prefix, zone.address, zone.slaves[zz], JSON.stringify(modbusRequestObject)));

                                this.modbusConnectorSendData(modbusRequestObject);

                                logger.info("%s_DRI[u:%02d] asynchronousCommit slave: %d, sent fan speed: %d, modbus register address: %d".sprintf(this.prefix, zone.address, zone.slaves[zz], zone.sent_fanspeed, hold + 1));
                                sleep(500);

                                logger.info("%s_DRI[u:%02d] slave: %d commit done <ambient: %.1f, setpoint: %.1f, power: %d, mode: %s( %d ), fan: %s, orientation: %s, tryes: %d, address: %d>".sprintf(this.prefix, zone.address, zone.slaves[zz], zone.mh_ambient, zone.mh_setpoint, zone.mh_power, HVAC_Driver.lg_modes[zone.sent_mode], zone.sent_mode, this.s_mh_fanspeeds[zone.sent_fanspeed], this.s_mh_orientations[sent_orientation], zone.dirty_counter, zone.slaves[zz]));
                            }
                        }
                        // -FEAT. HVAC Master/Slaves

                    } else {

                        logger.info("%s_DRI[u:%02d] commit skipped <ambient: %.1f, setpoint: %.1f, power: %d, mode: %s, fan: %s, orientation: %s>".sprintf(this.prefix, zone.address, zone.mh_ambient, zone.mh_setpoint, zone.mh_power, HVAC_Driver.lg_modes[zone.sent_mode], this.s_mh_fanspeeds[zone.sent_fanspeed], this.s_mh_orientations[sent_orientation]));
                    }
                }
            }

            this.addToDirty(-1);
            this.incommit = 0;
        }

        /* Supports a StationID != 1, add identifier value in XML file to set it */
        this.setIdentifier = function (id) {
            if (parseInt(id, 10) > 0) this.station = parseInt(id, 10);
        }

        this.setType = function (_type) {
            this.interfacetype = _type;
            logger.debug("%s_DRI[---] Set Type %s".sprintf(this.prefix, _type));
        }


        /*
		 *		param type is composed by two values: [ READ -> 1, WRITE -> 2] and [ COIL -> 0, COILS -> 1, REGISTER -> 2, REGISTERS -> 3, INPUT_REGISTER -> 4, INPUT_REGISTERS -> 5, STATUS -> 6, STATUSES -> 7 ]
		 *		for example: type = 21 stands for write coils 
		 */
        this.asynchronousFetchAll = function () {

            if (this.modbusConnectorConnectionStatus == 0) {
                logger.info("%s_DRI[---] modbus connector not connected, asynchronousFetchAll skipped".sprintf(this.prefix));
                return;
            }

            logger.info("%s_DRI[---] asynchronousFetchAll started".sprintf(this.prefix));

            for (var i in this.zones) {

                var zone = this.zones[i];

                var zoneSynchronizationArray = [];
                zoneSynchronizationArray['coil'] = 0;
                zoneSynchronizationArray['registers'] = 0;

                this.synchronizationArray[zone.id] = zoneSynchronizationArray;

                zone.dirty = false;

                var regcoil = 16 * (zone.address);
                var regholdings = 16 * (zone.address);

                logger.info("%s_DRI[u:%02d] asynchronousFetchAll fetching hvac unit parameters".sprintf(this.prefix, zone.address));

                var modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "read", cmdDevice: this.station, cmdString: "coil" }, params: { zone: zone.address, type: 10, address: regcoil, time_ms: Date.now() } };

                this.modbusConnectorSendData(modbusRequestObject);

                logger.info("%s_DRI[u:%02d] asynchronousFetchAll sending request to read coil power status, modbus address <%d>".sprintf(this.prefix, zone.address, regcoil));

                //sleep( 200 );

                var modbusRequestObject = { cmd: { cmdType: "modbus", cmdGoal: "read", cmdDevice: this.station, cmdString: "registers" }, params: { zone: zone.address, type: 13, address: regholdings, registers: 7, time_ms: Date.now() } };

                this.modbusConnectorSendData(modbusRequestObject);

                logger.info("%s_DRI[u:%02d] asynchronousFetchAll sending request to read registers status, modbus address <%d>".sprintf(this.prefix, zone.address, regcoil));

                //sleep( 200 );
            }

            logger.info("%s_DRI[---] asynchronousFetchAll asynchronousFetchAll end".sprintf(this.prefix));
        }

        this.setModbusConnectorConnectionStatus = function (connectionState) {

            var oldValue = this.modbusConnectorConnectionStatus;
            this.modbusConnectorConnectionStatus = connectionState;
            logger.info("%s_DRI[---] setModbusConnectorConnectionStatus modbusConnectorConnectionStatus: %d -> %d".sprintf(this.prefix, oldValue, connectionState));
        }


        /*
		 *		param type is composed by two values: [ READ -> 1, WRITE -> 2] and [ COIL -> 0, COILS -> 1, REGISTER -> 2, REGISTERS -> 3, INPUT_REGISTER -> 4, INPUT_REGISTERS -> 5, STATUS -> 6, STATUSES -> 7 ]
		 *		for example: type = 21 stands for write coils
		 *		response structure:
		 *		{
		 *			"request" = [ "read coil", "read coils", "read register", ... ],
		 *			"status" = [ 0 -> OK, 3000 -> ERROR ]
		 *			"msg", 
		 *			"payload" 
		 *		}
		 */
        this.parseResponseData = function (requestData, responseData) {

            logger.debug("%s_DRI[---] parseResponseData start".sprintf(this.prefix));

            var params = requestData.params;
            var hvacZoneAddress = params.zone;
            var type = params.type;
            var modbusAddress = parseInt(params.address, 10);
            var payload = responseData.payload;
            var status = parseInt(responseData.status, 10);

            logger.info("%s_DRI[---] parseResponseData hvac unit address: %d, type: %d, modbus address: %d, status: %d, payload: %s".sprintf(this.prefix, hvacZoneAddress, type, modbusAddress, status, JSON.stringify(payload)));

            // set readed value for hvac unit
            for (var i in this.zones) {

                var zone = this.zones[i];

                if (parseInt(zone.address, 10) == parseInt(hvacZoneAddress, 10)) {

                    //logger.info( "%s_DRI[u:%02d] parseResponseData hvac unit found".sprintf( this.prefix, zone.address ) );

                    switch (type) {

                        case 10:
                            {
                                // read on/off coil
                                //logger.info( "%s_DRI[u:%02d] parseResponseData case 10".sprintf( this.prefix, zone.address ) );

                                var powerstatus = parseInt(payload, 10);

                                if (powerstatus == null || typeof (powerstatus) == 'undefined' || isNaN(powerstatus)) {
                                    zone.hv_exist = false;
                                    continue;
                                } else {

                                    logger.info("%s_DRI[u:%02d] parseResponseData powerstatus: %d".sprintf(this.prefix, zone.address, powerstatus));

                                    zone.hv_exist = true;
                                    zone.hv_power = parseInt(powerstatus, 10);
                                }

                                if (zone.hv_power != zone.mh_power) {
                                    zone.dirty = true;
                                    logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by power: %d -> %d".sprintf(this.prefix, zone.address, zone.hv_power ? "ON" : "OFF", zone.mh_power ? "ON" : "OFF"));
                                }

                                this.synchronizationArray[zone.id]['coil'] = 1;

                                var go = true;	// used to check if either coil and registers have been readed

                                // check if both conditions are true:
                                // this.synchronizationArray[zone.id]['registers'] = 1
                                // this.synchronizationArray[zone.id]['coil'] = 1
                                // we do not know the exact order in which conditions will be read, may be coil first and then registers or vice versa,
                                // so make sure that the following operations are performed only when both ( coil and registers ) readings have been made
                                for (var key in this.synchronizationArray[zone.id]) {

                                    if (this.synchronizationArray[zone.id].hasOwnProperty(key)) {

                                        //logger.info( "%s_DRI[u:%02d] this.synchronizationArray[%d][%s]: %d".sprintf( this.prefix, zone.address, zone.id, key, this.synchronizationArray[zone.id][key] ) );
                                        if (this.synchronizationArray[zone.id][key] == 0) {
                                            go = false;
                                        }
                                    }
                                }

                                //logger.info( "%s_DRI[u:%02d] go: %s".sprintf( this.prefix, zone.address, go ? "true" : "false" ) );

                                if (go) {

                                    if (this.onAmbientCallback != null) {
                                        this.onAmbientCallback(zone.id);
                                    }

                                    if (zone.dirty) {
                                        logger.info("%s_DRI[u:%02d] parseResponseData zone is dirty ambient: %.1f, setpoint: %.1f, power: %d, mode: %s, fan: %s, orientation: %s, alarm: %d".sprintf(this.prefix, zone.address, zone.hv_ambient, zone.hv_setpoint, zone.hv_power, HVAC_Driver.s_modes[zone.hv_mode], this.s_mh_fanspeeds[zone.hv_fanspeed], this.s_mh_orientations[zone.hv_orientation], zone.hv_alarm));
                                        //if( !this.incommit ) this.commit( zone.id ); /* Resend commit for this zone */
                                        //logger.info( "%s_DRI[u:%02d] go to next zone".sprintf( this.prefix,zone.address ) );
                                        this.addToDirty(zone.id); /* Do not commit it immediatly but sent to dirty commit procedure */
                                        this.commitTimer.start(); /* delay the commit because when commit start, this loop stops */
                                    } else {
                                        logger.info("%s_DRI[u:%02d] parseResponseData zone is ok ambient: %.1f, setpoint: %.1f, power: %d, mode: %s, fan: %s, orientation: %s, alarm: %d".sprintf(this.prefix, zone.address, zone.hv_ambient, zone.hv_setpoint, zone.hv_power, HVAC_Driver.s_modes[zone.hv_mode], this.s_mh_fanspeeds[zone.hv_fanspeed], this.s_mh_orientations[zone.hv_orientation], zone.hv_alarm));
                                    }
                                }
                            }
                            break;

                        case 11:
                            {

                            }
                            break;

                        case 12:
                            {

                            }
                            break;

                        case 13:
                            {
                                //logger.info( "%s_DRI[u:%02d] parseResponseData case 13".sprintf( this.prefix, zone.address ) );

                                var status = payload;
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, registers: %s".sprintf( this.prefix, zone.address, address, status ) );
                                logger.info("%s_DRI[u:%02d] parseResponseData modbus address: %d, registers: %s".sprintf(this.prefix, zone.address, modbusAddress, status));

                                //for( var i = 0; i < 7; i++ ) {
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, register[%d]: %s".sprintf( this.prefix, zone.address, address, i, status[i] ) );
                                //}

                                var operationMode = parseInt(status[0], 10);
                                var fanSpeed = parseInt(status[1], 10);
                                var setPoint = parseInt(status[2], 10);
                                var tempUpperLimit = parseInt(status[3], 10);
                                var tempLowerLimit = parseInt(status[4], 10);
                                var ambientTemperature = parseInt(status[5], 10);
                                var errorCode = parseInt(status[6], 10);

                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit operation mode: %s ( %d ), modbus address <%d>".sprintf(this.prefix, zone.address, HVAC_Driver.lg_modes[operationMode], operationMode, modbusAddress));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit fan speed: %d, modbus address <%d>".sprintf(this.prefix, zone.address, fanSpeed, modbusAddress + 1));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set point: %.1f, modbus address <%d>".sprintf(this.prefix, zone.address, setPoint, modbusAddress + 2));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit temperature upper limit: %d, modbus address <%d>".sprintf(this.prefix, zone.address, tempUpperLimit, modbusAddress + 3));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit temperature lower limit: %d, modbus address <%d>".sprintf(this.prefix, zone.address, tempLowerLimit, modbusAddress + 4));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit ambient temperature: %d, modbus address <%d>".sprintf(this.prefix, zone.address, ambientTemperature, modbusAddress + 5));
                                logger.info("%s_DRI[u:%02d] parseResponseData hvac unit error code: %d, modbus address <%d>".sprintf(this.prefix, zone.address, errorCode, modbusAddress + 6));

                                switch (operationMode) {
                                    case 1: zone.hv_mode = mhOWN.OWN_THERMO_COOLING; break;
                                    case 2: zone.hv_mode = HVAC_Driver.OWN_THERMO_DRY; break;
                                    case 3: zone.hv_mode = HVAC_Driver.OWN_THERMO_FAN; break;
                                    case 4: zone.hv_mode = HVAC_Driver.OWN_THERMO_AUTO; break;
                                    case 5: zone.hv_mode = mhOWN.OWN_THERMO_HEATING; break;
                                }

                                zone.hv_fanspeed = fanSpeed;
                                zone.hv_orientation = -1;
                                zone.hv_setpoint = setPoint;
                                zone.hv_ambient = ambientTemperature;
                                zone.hv_temperature_upper_limit = tempUpperLimit;
                                zone.hv_temperature_lower_limit = tempLowerLimit;

                                if (operationMode != zone.sent_mode && zone.mh_power == true) {
                                    zone.dirty = true;
                                    logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by operation mode: %d -> %d".sprintf(this.prefix, zone.address, zone.sent_mode, operationMode));
                                }

                                if (zone.hv_fanspeed != zone.mh_fanspeed) {
                                    zone.dirty = true;
                                    logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by fan speed: %d -> %d".sprintf(this.prefix, zone.address, zone.hv_fanspeed, zone.mh_fanspeed));
                                }

                                if (zone.hv_power != zone.mh_power) {
                                    zone.dirty = true;
                                    logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by power: %d -> %d".sprintf(this.prefix, zone.address, zone.hv_power, zone.mh_power));
                                }

                                if (zone.hv_setpoint != zone.mh_setpoint) {
                                    zone.dirty = true;
                                    logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by set point: %.1f -> %.1f".sprintf(this.prefix, zone.address, zone.hv_setpoint, zone.mh_setpoint));
                                }

                                if (zone.hv_orientation != zone.mh_orientation && zone.mh_orientation != -1 && zone.hv_orientation != -1) {
                                    //zone.dirty = true;
                                    //logger.info("%s_DRI[u:%02d] parseResponseData hvac unit set dirty by orientation: %d -> %d".sprintf(this.prefix, zone.address, zone.hv_orientation, zone.mh_orientation));
                                }

                                // originale -> zone.hv_alarm = 0;
                                zone.hv_alarm = errorCode;

                                if (zone.hv_alarm != 0) {
                                    logger.warning("%s_DRI[u:%02d] parseResponseData WARNING!!!!! hvac unit malfunction!! error code: %d".sprintf(this.prefix, zone.address, zone.hv_alarm));
                                }

                                if (isNaN(zone.hv_setpoint)) {
                                    zone.dirty_counter = 0;
                                    zone.dirty = true;
                                }

                                this.synchronizationArray[zone.id]['registers'] = 1;

                                var go = true;	// used to check if either coil and registers have been readed

                                // check if both conditions are true:
                                // this.synchronizationArray[zone.id]['registers'] = 1
                                // this.synchronizationArray[zone.id]['coil'] = 1
                                // we do not know the exact order in which conditions will be read, may be coil first and then registers or vice versa,
                                // so make sure that the following operations are performed only when both ( coil and registers ) readings have been made
                                for (var key in this.synchronizationArray[zone.id]) {
                                    if (this.synchronizationArray[zone.id].hasOwnProperty(key)) {

                                        //logger.info( "%s_DRI[u:%02d] this.synchronizationArray[%d][%s] %d".sprintf( this.prefix, zone.address, zone.id, key, this.synchronizationArray[zone.id][key] ) );
                                        if (this.synchronizationArray[zone.id][key] == 0) {
                                            go = false;
                                        }
                                    }
                                }

                                //logger.info( "%s_DRI[u:%02d] go: %s".sprintf( this.prefix, zone.address, go ? "true" : "false" ) );

                                if (go) {

                                    if (this.onAmbientCallback != null) this.onAmbientCallback(zone.id);

                                    if (zone.dirty) {
                                        logger.info("%s_DRI[u:%02d] parseResponseData zone is dirty ambient: %.1f, setpoint: %.1f, power: %d, mode: %s, fan: %s, orientation: %s, alarm: %d".sprintf(this.prefix, zone.address, zone.hv_ambient, zone.hv_setpoint, zone.hv_power, HVAC_Driver.s_modes[zone.hv_mode], this.s_mh_fanspeeds[zone.hv_fanspeed], this.s_mh_orientations[zone.hv_orientation], zone.hv_alarm));
                                        //if( !this.incommit ) this.commit( zone.id ); /* Resend commit for this zone */
                                        //logger.info( "%s_DRI[u:%02d] go to next zone".sprintf( this.prefix,zone.address ) );
                                        this.addToDirty(zone.id); /* Do not commit it immediatly but sent to dirty commit procedure */
                                        this.commitTimer.start(); /* DMHITACHIM-1: delay the commit because when commit start, this loop stops */
                                    } else {
                                        logger.info("%s_DRI[u:%02d] parseResponseData zone is ok ambient: %.1f, setpoint: %.1f, power: %d, mode: %s, fan: %s, orientation: %s, alarm: %d".sprintf(this.prefix, zone.address, zone.hv_ambient, zone.hv_setpoint, zone.hv_power, HVAC_Driver.s_modes[zone.hv_mode], this.s_mh_fanspeeds[zone.hv_fanspeed], this.s_mh_orientations[zone.hv_orientation], zone.hv_alarm));
                                    }
                                }
                            }
                            break;

                        case 14:
                            {

                            }
                            break;

                        case 15:
                            {

                            }
                            break;

                        case 16:
                            {

                            }
                            break;

                        case 17:
                            {

                            }
                            break;

                        case 20:
                            {
                                var status = payload;
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, write coil response: %d".sprintf( this.prefix, zone.address, modbusAddress, status ) );
                            }
                            break;

                        case 21:
                            {
                                var status = payload;
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, write coils response: %d".sprintf( this.prefix, zone.address, modbusAddress, status ) );
                            }
                            break;

                        case 22:
                            {
                                var status = payload;
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, write register response: %d".sprintf( this.prefix, zone.address, modbusAddress, status ) );
                            }
                            break;

                        case 23:
                            {
                                var status = payload;
                                //logger.info( "%s_DRI[u:%02d] modbus address: %d, write registers response: %d".sprintf( this.prefix, zone.address, modbusAddress, status ) );
                            }
                            break;
                    }
                }
            }

            logger.debug("%s_DRI[---] parseResponseData end".sprintf(this.prefix));
        }

        this.endOfConfig = function () {

            //if( driverManagerVersionFlag ) {
            //	this.asynchronousFetchAll();
            //} else {
            //	this.fetchAll();
            //}

            this.asynchronousFetchAll();
        }

        // ------------------------------------------------------------------------
        // IP Details
        // ------------------------------------------------------------------------

        this.getIPAddress = function () {
            return this.ipaddress;
        }

        this.setIPAddress = function (ipaddress) {
            this.ipaddress = ipaddress;

            // if( !driverManagerVersionFlag ) {
            // logger.info( "%s_DRI[---] modbus client connect to ip address: %s".sprintf( this.prefix, this.ipaddress ) );
            // this.conn.connectTo( this.ipaddress );
            // }
            //logger.debug( "%s_DRI[---] Set IP address %s".sprintf( this.prefix,this.ipaddress ) );
            logger.info("%s_DRI[---] Set IP address %s".sprintf(this.prefix, this.ipaddress));
        }

        this.getPort = function () {
            return this.port;
        }

        this.setPort = function (port) {
            this.port = port;
            logger.debug("%s_DRI[---] Set IP port %s".sprintf(this.prefix, this.port));
        }

        this.setDebug = function (_debug) {

            this.debug = _debug;
            logger.debug("%s_DRI[---] set debug: %s".sprintf(this.prefix, this.debug));

            var modbusRequestObject = { cmd: { cmdType: "log", cmdGoal: "set", cmdDevice: "logger", cmdString: _debug }, params: { type: "log", time_ms: Date.now() } };

            this.modbusConnectorSendData(modbusRequestObject);
        }

        this.getDebug = function () {
            return this.debug;
        }

        // ------------------------------------------------------------------------
        // Zone setup
        // ------------------------------------------------------------------------

        /* Add the Zone identified by "id" and if auto fan speed is supported */
        this.addZone = function (id, address, slaves, autofan) {

            var zone = {};

            zone.address = parseInt(address, 10);            /* bus address, 0-31 that means 00:00..00:15 and 01:00..01:15 */
            zone.slaves = slaves;
            zone.autofan = autofan;                         /* auto fanspeed support */
            zone.id = id;
            zone.mh_power = false;                          /* setting: power state, off by default */
            zone.mh_setpoint = -1;                          /* setting: setpoint */
            zone.mh_mode = -1;                              /* setting: operation mode */
            zone.mh_ambient = -1;                           /* myhome state: ambient temperature */
            zone.mh_fanspeed = 0;                           /* setting: fan speed */
            zone.mh_orientation = -1;                       /* settings: orientation, tbd */
            zone.dirty_counter = 0;                         /* dirty flag */
            zone.hv_ambient = -1;                           /* hvac state: ambient temperature */
            zone.hv_mode = -1;
            zone.hv_fanspeed = -1;
            zone.hv_orientation = -1;
            zone.hv_setpoint = -1;
            zone.hv_temperature_upper_limit = -1;
            zone.hv_temperature_lower_limit = -1;
            zone.mh_forces = HVAC_Driver.REACHED_NONE;      /* Forces */
            zone.hv_type = 0;                               /* Type of interface, if required */
            zone.hv_exist = false;                          /* Zone-exists flag */
            zone.sent_mode = -1;
            zone.sent_fanspeed = -1;
            zone.hv_alarm = 0;
            zone.hv_filter = false;
            zone.dirty = false;

            this.zones.push(zone);

            logger.info("%s_DRI[u:%02d] Zone added zone id: %d, autofan: %d".sprintf(this.prefix, zone.address, zone.id, zone.autofan));
        }

        /* Specify the interface type of this unit */
        this.setZoneType = function (id, type) {
            i = this.getById(id);
            if (i != -1) {
                this.zones[i].hv_type = type;
                logger.info("%s_DRI[u:%02d] Zone type set as %d".sprintf(this.prefix, this.zones[i].address, type));
            }
        }

        /* Get store ambient temperature ( provided by device or from myhome ) */
        this.getAmbient = function (id) {
            i = this.getById(id);
            if (i != -1) {
                return this.zones[i].hv_ambient;
            }
        }

        /* Sets the My Home temperature */
        this.setAmbient = function (id, temperature) {
            i = this.getById(id);
            if (i != -1) {
                /* If this device does not provide ambient temperature, set this.hv_ambient also */
                if (this.zones[i].mh_ambient != temperature) {

                    logger.info("%s_DRI[u:%02d] mh_ambient NEW VALUE: %.1f -> %.1f".sprintf(this.prefix, this.zones[i].address, this.zones[i].mh_ambient, temperature));
                    this.zones[i].mh_ambient = temperature;
                    // this.commitTimer.start();
                    // this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        /* Sets the setpoint */
        this.setSetpoint = function (id, temperature) {

            i = this.getById(id);
            if (i != -1) {

                logger.info("%s_DRI[u:%02d] setSetpoint new set point value: %.1f".sprintf(this.prefix, this.zones[i].address, temperature));

                /* Fix the setpoint when save it because hvac supports integer value only */
                switch (this.zones[i].mh_mode) {

                    case mhOWN.OWN_THERMO_HEATING:
                        /* Ceil setpoint on heating */
                        temperature = Math.ceil(temperature);
                        break;

                    case mhOWN.OWN_THERMO_COOLING:
                        /* Floor setpoint on cooling */
                        temperature = Math.floor(temperature);
                        break;

                    default:
                        temperature = Math.round(temperature);
                        break;
                }

                logger.info("%s_DRI[u:%02d] setSetpoint set point value after rounding: %d".sprintf(this.prefix, this.zones[i].address, temperature));

                if (this.zones[i].mh_setpoint != temperature) {

                    logger.info("%s_DRI[u:%02d] mh_setpoint NEW VALUE: %.1f -> %.1f".sprintf(this.prefix, this.zones[i].address, this.zones[i].mh_setpoint, temperature));

                    this.zones[i].dirty_counter = 0;
                    this.zones[i].mh_setpoint = temperature;
                    this.addToDirty(id);
                    this.commitTimer.start();
                    this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        /* Gets stored setpoint */
        this.getSetpoint = function (id) {
            i = this.getById(id);
            if (i != -1) {
                return this.zones[i].mh_setpoint;
            }
            return -1;
        }

        /* Sets the operation mode, HEAT ( 0 ) or COOL( 1 ) */
        this.setOperationMode = function (id, mode) {

            i = this.getById(id);
            if (i != -1) {

                if (this.zones[i].mh_mode != mode) {

                    logger.info("%s_DRI[u:%02d] mh_mode NEW VALUE: %s (%d) -> %s (%d)".sprintf(this.prefix, this.zones[i].address, HVAC_Driver.modesDescription[this.zones[i].mh_mode], this.zones[i].mh_mode, HVAC_Driver.modesDescription[mode], mode));

                    this.zones[i].dirty_counter = 0;
                    this.zones[i].mh_mode = mode;
                    this.addToDirty(id);
                    this.commitTimer.start();
                    this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        this.getHWPower = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_power; } }
        this.getHWFanspeed = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_fanspeed; } }
        this.getHWSetpoint = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_setpoint; } }
        this.getHWOrientation = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_orientation; } }
        this.getHWOperationMode = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_mode; } }
        this.getHWAlarm = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_alarm; } }
        this.getHWFilter = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_filter; } }
        this.getHWExist = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].hv_exist; } }
        this.getHWExtra = function (id) { var i = this.getById(id); if (i != -1) { return this.zones[i].hv_extra; } }

        this.getPower = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].mh_power; } }
        this.getFanSpeed = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].mh_fanspeed; } }
        this.getSetpoint = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].mh_setpoint; } }
        this.getOrientation = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].mh_orientation; } }
        this.getOperationMode = function (id) { i = this.getById(id); if (i != -1) { return this.zones[i].mh_mode; } }

        this.isConnected = function () {

            for (var i in this.zones) {

                /* original code
				if( this.zones[i].hv_exist == 1 ) {
					return true;
				}
				*/

                logger.info("%s_DRI[u:%02d] isConnected, zone exist: %d, modbus connector connection: %d ".sprintf(this.prefix, this.zones[i].address, this.zones[i].hv_exist, this.modbusConnectorConnectionStatus));
                if (this.zones[i].hv_exist == 1 && this.modbusConnectorConnectionStatus == 1) {
                    return true;
                }
            }
            // original code -> return this.connected; 
            return false;
        }

        this.setPower = function (id, on) {

            i = this.getById(id);
            if (i != -1) {

                if (this.zones[i].mh_power != on) {

                    logger.info("%s_DRI[u:%02d] mh_power NEW VALUE: %s -> %s".sprintf(this.prefix, this.zones[i].address, this.zones[i].mh_power ? "ON" : "OFF", on ? "ON" : "OFF"));

                    this.zones[i].dirty_counter = 0;
                    this.zones[i].mh_power = on;
                    this.addToDirty(id);
                    this.commitTimer.start();
                    this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        /* Set the air orientation */
        this.setOrientation = function (id, value) {

            i = this.getById(id);
            if (i != -1) {

                if (this.zones[i].mh_orientation != value) {

                    logger.info("%s_DRI[u:%02d] mh_orientation NEW VALUE: %d -> %d".sprintf(this.prefix, this.zones[i].address, this.zones[i].mh_orientation, value));

                    this.zones[i].dirty_counter = 0;
                    this.zones[i].mh_orientation = value;
                    this.addToDirty(id);
                    this.commitTimer.start();
                    this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        /* Sets the fan speed, 0->auto, 1,2,3 speed, -1 off. */
        this.setFanSpeed = function (id, speed) {

            i = this.getById(id);
            if (i != -1) {

                if (this.zones[i].mh_fanspeed != speed) {

                    logger.info("%s_DRI[u:%02d] mh_fanspeed NEW VALUE: %s (%d) -> %s (%d)".sprintf(this.prefix, this.zones[i].address, HVAC_Driver.s_fanspeeds[this.zones[i].mh_fanspeed], this.zones[i].mh_fanspeed, HVAC_Driver.s_fanspeeds[speed], speed));

                    this.zones[i].dirty_counter = 0;
                    this.zones[i].mh_fanspeed = speed;
                    this.addToDirty(id);
                    this.commitTimer.start();
                    this.periodicRead.start(); /* Delay read job that can overwrite my values */
                }
            }
        }

        this.getById = function (id) {

            for (var i in this.zones) {

                if (this.zones[i].id == id) {
                    return i;
                }
            }
            return -1;
        }

        /*
		this.modbusError = function( a, b, c ) {
			logger.error( "Modbus error ( %d,%d,%d )".sprintf( a, b, c ) );
			this.connected = false;
		}
		
		this.modbusSucess = function( a, b ) {
			this.connected = true;
		}
		*/

        this.commitTimer = new mhTimer();
        this.commitTimer.setInterval(2000, true); /* Global commit wait interval, single shot timer */
        // ritardato per prova this.commitTimer.setInterval( 15000, true ); /* Global commit wait interval, single shot timer */
        // this.commitTimer.setAction( function() { if( !that.incommit ) that.commit( -2 ); } );
        this.commitTimer.setAction(this.asynchronousCommit.bind(this, -2));

        this.periodicRead = new mhTimer();
        this.periodicRead.setInterval(60000);
        this.periodicRead.setAction(this.asynchronousFetchAll.bind(this));

        this.periodicRead.start();
    }

    return drv;
})();