// This class connects MH to generic HVAC driver

var HVAC_Driver = (function() {

    /* _zones is the array of zones, with
     * .ownserver -> id_own
     * .myhome    -> myhome ZAZB
     * .group     -> daikin unit id
     * .setpoint  -> current setpoint (from mh)
     * .ambient   -> ambient temperature (from daac)
     * .mhambient -> ambient temperature (from myhome)
     * .fanspeed  -> current fan speed preference (from mh, 0 to 3)
     * .standby   -> software standby status (OFF or w/fan speed auto override while sending status to DAAC) when setpoint is reached
     * .func      -> probe model (HOTEL or MYHOME)
     * .ledstatus -> cache last led status to avoid continuous led update
     *
     * This driver can manage this kind of scenario:
     * - Single Probe or Single Probe and 3550 central unit
     * - Single Probe or Single Probe and 3550 central unit + Bacnet TS (configure TS to manage only SETPOINT, SPEED and DIRECTION profile)
     * - Bacnet TS and use HVAC system as ambient reference (without MH probe!)
     */

    // constructor
    var drv = function() {
    
        that = this;

        //+FEAT. Er2 Management
        var forcedLedUpdate = {};
        //-FEAT. Er2 Management
    
        // External constants
        this.DRV_LOCK_NO = 0;
        this.DRV_LOCK_YES = 1;
        this.DRV_LOCK_TILL_SETPOINTCHANGE = 2;
    
        // Constants
        this.heat_fandelta = 3.0;                         // Celsius degree from setpoint to 'dim' the fan speed
        this.heat_integrate_delta = 1.0;                  // Setpoint difference between radiant and fan system in integrated mode
                  
        this.cool_fandelta = 3.0;                         // Celsius degree from setpoint to 'dim' the fan speed
        this.cool_integrate_delta = 1.0;                  // Setpoint difference between radiant and fan system in integrated mode

        this.heat_hysteresis = 0.2;
        this.cool_hysteresis = 0.2;
                  
        // Preferences              
        this.mh_as_reference = true;                      // use MH ambient or HVAC ambient temperature as reference (ignored if probes are not installed)
        this.off_thermal_protection = false;               // Turn off system on thermal protection or send the setpoint?
        
        this.when_reached_heat = HVAC_Driver.REACHED_FAN_LOW;  // What happen when setpoint is reached: OFF, FAN LOW/AUTO or NONE?
        this.when_reached_cool = HVAC_Driver.REACHED_FAN_LOW;  // What happen when setpoint is reached: OFF, FAN LOW/AUTO or NONE?
    
        // Variables
        this.zones = [];
        this.debug = false;
    
        this.bacnetport = 20020;
        this.bacnetserver;
        this.hvac; /* adapter */
        this.name = "Generic HVAC Adapter";
        this.prefix = "MH";
        this.zoneid = 0;
    
        this._bkmodes = [-1,mhOWN.OWN_THERMO_HEATING,mhOWN.OWN_THERMO_COOLING,HVAC_Driver.OWN_THERMO_FAN,HVAC_Driver.OWN_THERMO_DRY,HVAC_Driver.OWN_THERMO_AUTO,HVAC_Driver.OWN_THERMO_AUTO,HVAC_Driver.OWN_THERMO_AUTO];
  
        this.sharedTimer = new mhTimer();
        this.sharedTimer.setInterval( 4000, true ); /* Global commit wait interval, single shot timer */
        this.sharedTimer.setAction(function() { that.writeOnShared(-1); });
  
        shared.write( 'zones', JSON.stringify({}) );
        shared.write( 'gateway', JSON.stringify({}) );
	
        this.setHttpConnectorParameters = function() {
            this.hvac.setHttpConnectorParameters();
        }
	
        this.parseResponseData = function( req, resp ) {
            this.hvac.parseResponseData( req, resp );
        }
	
        this.setModbusConnectorConnectionStatus = function( connectionState ) {
            this.hvac.setModbusConnectorConnectionStatus( connectionState );
        }
  
        // ------------------------------------------------------------------------
        // Zone setup
        // ------------------------------------------------------------------------
        /* Add Zone and specifies address only and automatic-fan mode support from hvac */
        this.addZone = function (ownserver, address, bacnetarea, bacnetaddress, hvacstation, hvacaddress, hvacSlaveAddressArray, autofan) {

            try {

                this.zoneid++;

                var data = {};

                data.name = "";
                data.mh_ownserver = parseInt(ownserver, 10);     // OWN Server of the zone
                data.mh_address = parseInt(address, 10);         // Base address of the zones, 0 for bacnet-only
                data.bac_area = parseInt(bacnetarea, 10);        // Base address of the zones in bacnet mode, 0 for probe only
                data.bac_address = parseInt(bacnetaddress, 10);  // Base address of the zones in bacnet mode, 0 for probe only
                data.mh_fanprobe = false;                       // Fan mode probe
                data.hvac_address = String(hvacaddress);
                data.heat_actuators = 'PHY';                    // PHY->led status is controlled by probe / VIRT->led status is controlled by driver / NONE->bacnet only
                data.cool_actuators = 'PHY';                    // PHY->led status is controlled by probe / VIRT->led status is controlled by driver / NONE->bacnet only
                data.heat_mode = 'SINGLE';              
                data.cool_mode = 'SINGLE';    
                data.heat_adjust = false;                       // Adjust setpoint with probes diffs 
                data.cool_adjust = false;                       // Adjust setpoint with probes diffs 
                data.heat_airdirection = -1;                    // Air direction, or none 
                data.cool_airdirection = -1;                    // Air direction, or none 
                data.heat_virtualactuator = 1;
                data.cool_virtualactuator = 1;
          
                data.lock = this.DRV_LOCK_NO;                   // Lock mode, disable zone till unlock (1) or at setpoint change (2)
          
                data.mode = -1;
                data.mode_whenoff = -1;
                data.power = false;
                data.autofan = autofan;
                data.zoneid = this.zoneid;
                data.bac_canupdate = true;
                data.alarm = 0;                                 // Alarm flag
                data.extra = {};                                // Extra info that comes from the unit
                data.bac_alarm = 0;
                data.filter = false;                            // Filter flag
                data.bac_filter = 0;                       // Filter Error sended
          
                data.mh_ambient = -1;                           // Current ambient temperature
                data.mh_setpoint = -1;                          // Current setpoint (real, from mh)
                data.mh_setpoint_integrated = -1;               // Current setpoint (real, from mh + integration but without adjust -- needed by isSetpointReached and setFanSpeed when MH is the reference)
                data.mh_direction = -1;                         // Current air direction
                data.mh_fanspeed = 0;                           // Current fanspeed (0-3) or 0 if not-present
                data.led_status = -1;                           // Led status (0->off, 1->on)
                data.no_hysteresis = false;                     // Disable Hysteresis (when current is > 1 but new setpoint is lower than ambient temperature)
                data.setPointReached = false;
                data.hysteresis = false;
        
                data.lock_button_lock = -1;
                data.lock_button_unlock = -1;
                data.lock_button_temporary = -1;
                data.lock_cen_a = -1;
                data.lock_cen_pl = -1;
                data.lock_cen_bus = -1;
                data.lock_cenplus = -1;

                // +Better Er2 management
                data.startupLedStatusUpdate = true;             // This flag shows if the probe's led status has been updated at driver startup or not

                data.gatewayStatusUpdateTimer = new mhTimer();
                data.gatewayStatusUpdateTimer.setInterval(59 * 60 * 1000); // cyclic timer
                // only for debugging purpose -> data.gatewayStatusUpdateTimer.setInterval(2 * 60 * 1000); // cyclic timer
                data.gatewayStatusUpdateTimer.setAction(this.gatewayStatusUpdater.bind(this, data.zoneid));
                // -Better Er2 management
        
                this.zones.push(data);
        
                if (data.mh_address != 0) {
                    logger.info("%s_MHH[u:%s] new zone added with probe (gw:%02d, a:%d)".sprintf(this.prefix, data.hvac_address, data.mh_ownserver, data.mh_address));
                }

                if (data.bac_area != 0 && data.bac_address != 0) {
                    logger.info("%s_MHH[u:%s] new zone added with bacnet (ba:%d, a:%d)".sprintf(this.prefix, data.hvac_address, data.bac_area, data.bac_address));
                }

                this.hvac.addZone(data.zoneid, hvacaddress, hvacSlaveAddressArray, data.autofan);
        
                this.sharedTimer.start();

                return this.zones.length - 1;

            } catch (ex) {
                logger.error("%s_MHH[gw:%02d,z:%02d] error adding zone: %s".sprintf(this.prefix, ownserver, address, ex));
            }
        };

        //+FEAT. Er2 Management
        this.setForcedLedUpdate = function(zoneArrayIndex, _force) {
            forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = _force;
        }
	
        this.getForcedLedUpdate = function (zoneId) {
            //logger.debug("MHH[---] forcedLedUpdate[%d]: %d".sprintf(zoneId, forcedLedUpdate[zoneId]));
            var zoneArrayIndex = this.getZoneArrayIndexByZoneId(zoneId);

            if (zoneArrayIndex != -1) {	        
                logger.debug("%s_MHH[gw:%02d,z:%02d,u:%s] forcedLedUpdate: %d".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, forcedLedUpdate[zoneId]));
                return forcedLedUpdate[zoneId];
            }

            return false;
        }
        //-FEAT. Er2 Management
    
        /* Sets a global 'type' property */
        this.hvacSetType = function(_in) {
            if (typeof(this.hvac.setType) == 'function') {
                this.hvac.setType(_in);
            } else {
                logger.info("%s_MHH[---] HVAC Type <%s> set, but adapter does not support this parameter".sprintf(this.prefix, _in));
            }
        }

        /* Sets a global 'identifier' property */
        this.hvacSetIdentifier = function(_in) {
            if (typeof(this.hvac.setIdentifier) == 'function') {
                this.hvac.setIdentifier(_in);
            } else {
                logger.info("%s_MHH[---] HVAC Identifier <%s> set, but adapter does not support this parameter".sprintf(this.prefix, _in));
            }
        }

        this.getZoneArrayIndexByZoneId = function (zoneId) {
            //logger.info("+getZoneArrayIndexByZoneId -> zoneId: %d".sprintf( zoneId ));

            for (var i in this.zones) {
                //logger.info("this.zones[%d].zoneId: %d".sprintf( i, this.zones[i].zoneid ));
                if (this.zones[i].zoneid == zoneId) {
                    //logger.info("-getZoneArrayIndexByZoneId -> return: %d".sprintf( i ));
                    return i;
                }
            }

            //logger.info("-getZoneArrayIndexByZoneId -> return: -1");
            return -1;
        }
    
        this.endOfConfig = function() {

            try {
                this.hvac.endOfConfig();

                // +Better Er2 management
                this.startupLedStatusUpdateTimer.start();
                logger.info("%s_MHH[---] startupLedStatusUpdateTimer start".sprintf(this.prefix));
                // -Better Er2 management
            
            } catch (ex) {
                logger.error("%s_MHH[---] error: %s".sprintf(this.prefix, ex));
            }
        }

        // +Better Er2 management
        this.startupLedStatusUpdater = function () {

            //logger.debug("%s_MHH[---] startupLedStatusUpdater".sprintf(this.prefix));
            logger.info("%s_MHH[---] startupLedStatusUpdater".sprintf(this.prefix));

            var oneMoreTime = false;

            //loop over zones
            for (var zoneArrayIndex in this.zones) {

                //logger.debug("%s_MHH[HVAC unit:%s] startupLedStatusUpdater, startupLedStatusUpdate: %s".sprintf(this.prefix, this.zones[i].hvac_address, this.zones[i].startupLedStatusUpdate ? "TRUE" : "FALSE"));
                //logger.debug("%s_MHH[gw:%02d,z:%02d,u:%s] startupLedStatusUpdater, startupLedStatusUpdate: %s".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, this.zones[i].startupLedStatusUpdate ? "TRUE" : "FALSE"));
                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] startupLedStatusUpdater, startupLedStatusUpdate: %s".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, this.zones[zoneArrayIndex].startupLedStatusUpdate ? "TRUE" : "FALSE"));
                if (this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    //this.ledUpdate( this.zones[i].zoneid, true );
                    this.ledUpdate( zoneArrayIndex, true );
                }

                // if zone's led status update is not successfull, have to run the timer until all zones leds have been updated
                if (this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    oneMoreTime = true;
                }
            }

            if (oneMoreTime) {
                this.startupLedStatusUpdateTimer.start();
                //logger.debug("%s_MHH[---] startupLedStatusUpdateTimer start".sprintf(this.prefix));
                logger.info("%s_MHH[---] startupLedStatusUpdateTimer start".sprintf(this.prefix));
            } else {
                //logger.debug("%s_MHH[---] startupLedStatusUpdateTimer stop".sprintf(this.prefix));
                logger.info("%s_MHH[---] startupLedStatusUpdateTimer stop".sprintf(this.prefix));
            }
        }

        // update gateway status
        this.gatewayStatusUpdater = function (zoneId) {

            var zoneArrayIndex = this.getZoneArrayIndexByZoneId(zoneId)
            if (zoneArrayIndex != -1) {

                //logger.info("%s_MHH[HVAC unit:%s] +gatewayStatusUpdater".sprintf(this.prefix, this.zones[zoneArrayIndex].hvac_address));
                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] +gatewayStatusUpdater".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address));

                var hvacZoneExist = this.hvac.getHWExist(zoneId);
                if (!hvacZoneExist) {
                    logger.warning("%s_MHH[gw:%02d,z:%02d,u:%s] gatewayStatusUpdater, WARNING!! hvac zone does not exists, skipping led status update".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address));                 
                    return;
                }

                var fanSpeed;
                var mode;

                //mhOWN.ThermoZone_GatewayUpdate( <gateway id>, <my home zone address>, <fan speed>, <mode>, <set point> );

                // fan speed    hex      description
                // 0            00       Automatic
                // 1            01       Minimum fan speed
                // 2            02       Medium fan speed
                // 3            03       Maximum fan speed
                // 4            04       Silent mode
                // 13           0D       Fan-coil power on, heating
                // 14           0E       Fan-coil power on, cooling
                // 15           0F       Fan-coil power off, setpoint reached

                // mode     description
                // 0        Heating
                // 1        Cooling				
                // 2        Anti freeze protection
                // 3        Thermal protection
                // 4        Generic protection
                // 5        Fan
                // 6        Dry
                // 7        Automatic
                // 15       OFF

                if (this.zones[zoneArrayIndex].led_status) {
                    fanSpeed = 0;  // Automatic
                } else {
                    fanSpeed = 15;  // Off
                }

                switch (this.zones[zoneArrayIndex].mode) {

                    case mhOWN.OWN_THERMO_COOLING:  // 2

                        switch (this.zones[zoneArrayIndex].submode) {

                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                                mode = 1;       // Cooling
                                break;

                            case mhOWN.OWN_THERMO_AUTO:   // 11
                                mode = 1;       // Cooling
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14

                                // Thermal protection:
                                // fan speed: 15, hex: 0F, description: fan-coil power off, setpoint reached
                                // mode: 3, description: Thermal protection
                                mode = 3;       // Thermal protection
                                break;

                            default:
                                mode = 1;       // Cooling
                        }
                        break;

                    case mhOWN.OWN_THERMO_HEATING:  // 1

                        switch (this.zones[zoneArrayIndex].submode) {

                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                                mode = 0;       // Heating
                                break;

                            case mhOWN.OWN_THERMO_AUTO:   // 11
                                mode = 0;       // Heating
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14
                                // Anti freeze protection
                                // fan speed: 15, hex: 0F, description: fan-coil power off, setpoint reached
                                // mode:2, description: Anti freeze protection
                                mode = 2;       // Anti freeze protection
                                break;

                            default:
                                mode = 0;       // Heating
                        }
                        break;

                    case mhOWN.OWN_THERMO_OFF:  // 0
                        mode = 15;      // Off
                        fanSpeed = 15;  // Off
                        break;
                }

                //logger.info("%s_MHH[HVAC unit:%s] ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %d, mode: %d, set point: %.1f".sprintf(this.prefix, this.zones[zoneArrayIndex].hvac_address, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                //logger.info("%s_MHH[HVAC unit:%s] probe led status: %d".sprintf(this.prefix, this.zones[zoneArrayIndex].hvac_address, this.zones[zoneArrayIndex].led_status));
                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %d, mode: %d, set point: %.1f".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] probe led status: %d".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, this.zones[zoneArrayIndex].led_status));

                mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] -gatewayStatusUpdater".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address));

            } else {
                logger.warning("%s_MHH[---] gatewayStatusUpdater, Warning! got zoneArrayIndex = -1 from zone id: %d, skipping gateway update!!".sprintf(this.prefix, zoneId));
            }        
        }
        // -Better Er2 management	
       
        // Called by HVAC on every ambient temperature callback
        this.onAmbientCallback = function(i) {
            zoneArrayIndex = this.getZoneByZoneId(i);
            if (zoneArrayIndex == -1) return;
      
            if (this.zones[zoneArrayIndex].mh_address == 0) {
                this.hvac.setAmbient(i,this.hvac.getAmbient(i));
        
                /* Startup situation? Use HVAC as starting status because MH probe is not configured */
                if( this.zones[zoneArrayIndex].mode == -1 && this.hvac.getHWExist(i) ) { /* evita la valorizzazione se la zona non ne sa nulla */
                    this.setFanSpeed(zoneArrayIndex,this.hvac.getHWFanspeed(i));
                    this.setSetpoint(zoneArrayIndex,this.hvac.getHWSetpoint(i));
                    this.setOrientation(zoneArrayIndex,this.hvac.getHWOrientation(i));
                    if( this.hvac.getHWPower(i) == false ) {
                        this.setMode(zoneArrayIndex,mhOWN.OWN_THERMO_OFF,-1);
                    } else {
                        this.setMode(zoneArrayIndex,this.hvac.getHWOperationMode(i),-1);
                    }
          
                    // original -> logger.info( "%s_MHH[u:%s] retrieve cached status from HVAC unit < power: %d, mode: %d, set point: %.1f, fan speed: %d, orientation: %d>".sprintf( this.prefix, this.zones[zoneArrayIndex].hvac_address, this.zones[zoneArrayIndex].power, this.zones[zoneArrayIndex].mode, this.zones[zoneArrayIndex].mh_setpoint, this.zones[zoneArrayIndex].mh_fanspeed, this.zones[zoneArrayIndex].mh_direction ) );
                    logger.info( "%s_MHH[u:%s] retrieve cached status from HVAC unit < power: %d, mode: %d, set point: %.1f, fan speed: %d>".sprintf( this.prefix, this.zones[zoneArrayIndex].hvac_address, this.zones[zoneArrayIndex].power, this.zones[zoneArrayIndex].mode, this.zones[zoneArrayIndex].mh_setpoint, this.zones[zoneArrayIndex].mh_fanspeed ) );
                }
            }
      
            this.zones[zoneArrayIndex].extra = this.hvac.getHWExtra(i);
            this.zones[zoneArrayIndex].alarm = this.hvac.getHWAlarm(i);
            this.zones[zoneArrayIndex].filter = this.hvac.getHWFilter(i);
      
            /* If no orientation is set and a TS is configured with a probe, fetch status from HVAC */
            // if( this.zones[zoneArrayIndex].mh_direction == -1 && this.zones[zoneArrayIndex].bac_area != 0 && this.zones[zoneArrayIndex].bac_address != 0 ) {
            // this.setOrientation( zoneArrayIndex, this.hvac.getHWOrientation( i ) );
            // }
      
            /* 2015.03.13 - re-calc adjusted setpoint if needed */
            if( ( this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_adjust ) || ( this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_adjust ) ) {
                this.setSetpoint( zoneArrayIndex, this.zones[zoneArrayIndex].mh_setpoint ); /* Recommit current setpoint */
            }
      
            if( this.zones[zoneArrayIndex].mh_fanspeed == 0 ) {
                /* Recalc fan speed on auto!!! */
                this.setFanSpeed( zoneArrayIndex, this.zones[zoneArrayIndex].mh_fanspeed );
            }
      
            if( ( !this.mh_as_reference ) || this.zones[zoneArrayIndex].mh_address == 0 ) { /* if you want to use HVAC as referse or probe is missing */
                this.sendBacnetUpdate( zoneArrayIndex );
                this.forces( zoneArrayIndex );
            }
      
            if( this.zones[zoneArrayIndex].mh_address != 0 ) {
                this.ledUpdate( zoneArrayIndex, forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] );
            }
      
      
            /* If receive the ambient temperature and the setpoint is not available, please request it again */        
            if( this.zones[zoneArrayIndex].mh_address != 0 && this.zones[zoneArrayIndex].mh_setpoint == -1 ) {
                mhOWN.sendFrame( this.zones[zoneArrayIndex].mh_ownserver, format( "*#4*%d*12##", this.zones[zoneArrayIndex].mh_address ) );
            }
      
            this.sharedTimer.start();
        }
    
        this.editZone_Fanprobe = function(zoneArrayIndex,fanprobe) {
            if (zoneArrayIndex == -1) return;
            this.zones[zoneArrayIndex].mh_fanprobe = fanprobe;
        }

        this.editZone_HVAC_Type = function(zoneArrayIndex,type) {
            if (zoneArrayIndex == -1) return;
            this.hvac.setZoneType(this.zones[zoneArrayIndex].zoneid,type);
        }

        this.editZone_Name = function(zoneArrayIndex,name) {
            if (zoneArrayIndex == -1) return;
            this.zones[zoneArrayIndex].name = name;
        }
    
        this.editZone_AdjustMode = function( zoneArrayIndex, mode ) {

            if (zoneArrayIndex == -1) return;

            var logHeading = "%s_MHH[gw:%02d,z:%02d,u:%s]".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address);

            logger.info("%s editZone_AdjustMode mode: %s".sprintf(logHeading, mode));
      
            switch( String( mode ).toUpperCase() ) {

                case 'HEAT':
                case 'HEATING':
                    this.zones[zoneArrayIndex].heat_adjust = true;
                    break;
			
                case 'COOL':
                case 'COOLING':
                    this.zones[zoneArrayIndex].cool_adjust = true;
                    break;
			
                case 'BOTH':
                    this.zones[zoneArrayIndex].cool_adjust = true;
                    this.zones[zoneArrayIndex].heat_adjust = true;
                    break;
			
                case 'NONE':
                    this.zones[zoneArrayIndex].cool_adjust = false;
                    this.zones[zoneArrayIndex].heat_adjust = false;
                    break;
            }

            logger.info("%s editZone_AdjustMode heat adjust: %s, cool adjust: %s".sprintf(logHeading, this.zones[zoneArrayIndex].heat_adjust ? "TRUE" : "FALSE", this.zones[zoneArrayIndex].cool_adjust ? "TRUE" : "FALSE"));
        }
    
        this.editZone_Heating_Details = function(zoneArrayIndex,heat_actuators,heat_mode,heat_airdirection) {

            if (zoneArrayIndex == -1) return;
            this.zones[zoneArrayIndex].heat_actuators = heat_actuators;
            this.zones[zoneArrayIndex].heat_mode = heat_mode;
            this.zones[zoneArrayIndex].heat_airdirection = heat_airdirection;
            logger.debug("%s_MHH[u:%s] heating details added <actuators=%s, integration=%s, direction=%s>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,heat_actuators,heat_mode,heat_airdirection));
        }    
    
        this.editZone_Cooling_Details = function(zoneArrayIndex,cool_actuators,cool_mode,cool_airdirection) {

            if (zoneArrayIndex == -1) return;
            this.zones[zoneArrayIndex].cool_actuators = cool_actuators;
            this.zones[zoneArrayIndex].cool_mode = cool_mode;
            this.zones[zoneArrayIndex].cool_airdirection = cool_airdirection;
            logger.debug("%s_MHH[u:%s] cooling details added <actuators=%s, integration=%s, direction=%s>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,cool_actuators,cool_mode,cool_airdirection));
        }        
    
        this.editZone_Lock = function (zoneArrayIndex, cen, cenplus, btn_lock, btn_unlock, btn_temporary) {

            if (zoneArrayIndex == -1) return;
      
            this.zones[zoneArrayIndex].lock_button_lock = btn_lock;
            this.zones[zoneArrayIndex].lock_button_unlock = btn_unlock;
            this.zones[zoneArrayIndex].lock_button_temporary = btn_temporary;
      
            /* Split CEN */
            matches = cen.split("-");
            if (matches.length == 3) {
                this.zones[zoneArrayIndex].lock_cen_bus = parseInt(matches[0],10);
                this.zones[zoneArrayIndex].lock_cen_a = parseInt(matches[1],10);
                this.zones[zoneArrayIndex].lock_cen_pl = parseInt(matches[2],10);
                if (this.zones[zoneArrayIndex].lock_cen_bus >= 0 && this.zones[zoneArrayIndex].lock_cen_a > 0 && this.zones[zoneArrayIndex].lock_cen_pl > 0) {
                    logger.debug("%s_MHH[u:%s] locking details added <lock CEN=%d-%d-%d btn_lock=%d,btn_unlock=%d,btn_temporary=%d>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,
                      this.zones[zoneArrayIndex].lock_cen_bus, this.zones[zoneArrayIndex].lock_cen_a, this.zones[zoneArrayIndex].lock_cen_pl,
                      this.zones[zoneArrayIndex].lock_button_lock,this.zones[zoneArrayIndex].lock_button_unlock,this.zones[zoneArrayIndex].lock_button_temporary));
                }
            }
      
            this.zones[zoneArrayIndex].lock_cenplus = cenplus;
            if (this.zones[zoneArrayIndex].lock_cenplus > 0) {

                logger.debug("%s_MHH[u:%s] details added <lock CEN+=%d btn_lock=%d,btn_unlock=%d,btn_temporary=%d>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,
                    this.zones[zoneArrayIndex].lock_cenplus,
                    this.zones[zoneArrayIndex].lock_button_lock,this.zones[zoneArrayIndex].lock_button_unlock,this.zones[zoneArrayIndex].lock_button_temporary));
            }
        }

        this.evaluateZoneFanSpeedForProbe = function (zoneArrayIndex) {

            var fanSpeed;
            logger.debug("%s evaluateZoneFanSpeedForProbe led status: %d".sprintf(this.prefix, this.zones[zoneArrayIndex].led_status));

            if (this.zones[zoneArrayIndex].led_status == 0) {

                fanSpeed = 15;

            } else {

                fanSpeed = this.zones[zoneArrayIndex].mh_fanspeed;
            }

            logger.debug("%s evaluateZoneFanSpeedForProbe returning fan speed: %d".sprintf(this.prefix, fanSpeed));
            return fanSpeed;
        }
    
        /* Start the bacnet server */
        this.startBacnet = function(port) {
            this.bacnetport = port;
            this.bacnetserver = new mhOWNServer(this.bacnetport);
            this.bacnetserver.onFrameEvent(function(id,frame) {that.onBacnetFrame(id,frame)});
            this.bacnetserver.onConnectEvent(function(id,address,type) { 
                if (that.debug) logger.debug(format("%s_MHH[---] received bacnet connection from <%s>".sprintf(this.prefix,address)));
                if(type==1) { /* 1 MONITOR || 0 CLIENT */
                    for (var id in that.zones){
                        that.zones[id].bac_alarm=0;
                        that.zones[id].bac_filter=0;
                    }
                }
            });
        }
    
        this.onBacnetFrame = function(_id,frame) {
            if (match = frame.match(/\*#4\*#(\d+)#(\d+)\*51##/)) {
                id = this.getZoneByBacnet(match[1],match[2]);
                if (id == -1) return;
                logger.debug(format("%s_MHH[u:%s] received status request".sprintf(this.prefix,this.zones[id].hvac_address)));
                this.sendBacnetUpdate(id);
                return;
            }
            // *#4*#1#1*#51*255**1*0*1*##
            // *#4*<where>*#51*<RoomTempSetPoint>*<Status>*<WorkingMode>*<FanSpeed>*<AirDirection>*<FilterFault>##
            if (match = frame.match(/\*#4\*#(\d+)#(\d+)\*#51\*(\d*)\*(\d*)\*(\d*)\*(\d*)\*(\d*)\*(\d*)##/)) {
                id = this.getZoneByBacnet(match[1],match[2]);
                if (id == -1) return;
        
                this.zones[id].bac_canupdate = false;
                logger.debug(format("%s_MHH[u:%s] received set request <%s>".sprintf(this.prefix,this.zones[id].hvac_address,frame)));
                if (parseFloat(match[3]) > 0) { /* New Setpoint */
                    this.setSetpoint(id,parseFloat(match[3])/10);
                    if (this.zones[id].mh_address != 0) { /* If a real probe is configured */
                        mhOWN.ThermoZone_Setpoint(this.zones[id].mh_ownserver,this.zones[id].mh_address,this.zones[id].mh_setpoint);
                    }
                }
                if (match[6] != '') { /* New fanspeed in standard range 0,1,2,3 */
                    var fanspeed = parseInt(match[6],10);
                    if (fanspeed == 4) fanspeed=1; /* remap silent mode to slow */
                    this.setFanSpeed(id, fanspeed);
                    if (this.zones[id].mh_address != 0 && this.zones[id].mh_fanprobe) { /* If a real probe with fan speed selector is configured */
                        mhOWN.ThermoZone_FanSpeed(this.zones[id].mh_ownserver,this.zones[id].mh_address,this.zones[id].mh_fanspeed);
                    }
                }
        
                if (match[4] != '') { /* Operation mode (0->off, 1->on) */
                    if (parseInt(match[4],10) == 1) { /* On */
                        if (match[5] != '') {
                            /* Mode is sent by TS, so ignore ON command */
                        } else {
                            /* We need to retreive original mode and re-set it */
                            this.setMode(id,this.zones[id].mode,-1);
                            if (this.zones[id].mh_address != 0) {
                                mhOWN.sendFrame(this.zones[id].mh_ownserver,"*#4*%d*#14*%.0f*3##".sprintf(this.zones[id].mh_address,this.zones[id].mh_setpoint*10));
                            }
                        }
            
                        this.zones[id].power = true;
                        logger.debug( format( "%s_MHH[---] zone: %d power <%d>".sprintf( this.prefix, this.zones[id].mh_address, this.zones[id].power ) ) );
			
                    } else { /* Off */
                        this.setMode(id,mhOWN.OWN_THERMO_OFF,-1);
                        if (this.zones[id].mh_address != 0) {
                            mhOWN.sendFrame(this.zones[id].mh_ownserver,"*4*303*#%d##".sprintf(this.zones[id].mh_address));
                        }
                    }
                }
                if (match[7] != '') { /* New air direction/orientation */
                    var orientation = parseInt(match[7],10);
                    this.setOrientation(id,orientation);
                }               
        
                if (match[5] != '') { /* Working Mode */
                    switch (parseInt(match[5],10)) {
                        case 0: /* Inactive */
                            break;
                        case 1: /* Winter */
                            this.setMode(id,mhOWN.OWN_THERMO_HEATING,-1);
                            break;
                        case 2: /* Summer */
                            this.setMode(id,mhOWN.OWN_THERMO_COOLING,-1);
                            break;
                        case 3: /* Fan */
                            this.setMode(id,HVAC_Driver.OWN_THERMO_FAN,-1);
                            break;
                        case 4: /* Dry */
                            this.setMode(id,HVAC_Driver.OWN_THERMO_DRY,-1);
                            break;
                        case 5: /* NormalAuto */
                        case 6: /* FastAuto */
                        case 7: /* EnergySavingAuto */
                            this.setMode(id,HVAC_Driver.OWN_THERMO_AUTO,-1);
                            break;
            
                    }
                }

                this.zones[id].bac_canupdate = true;
                this.sendBacnetUpdate(id);
                return;
            }

            logger.debug(format("%s_MHH[---] received unmanaged bacnet frame <%s>".sprintf(this.prefix,frame)));
        }
    
        /* Send updated information to all connected clients about _id_ zone */
        this.sendBacnetUpdate = function (id) {

            if (id == -1) return;
            this.sharedTimer.start();
      
            if (this.zones[id].bac_area == 0 || this.zones[id].bac_address == 0) return;
      
            if (!this.zones[id].bac_canupdate) {

                logger.debug(format("%s_MHH[u:%s] sendBacnetUpdate skipped, disabled".sprintf(this.prefix,this.zones[id].hvac_address)));
                return;
            } 
      
            var ambient = this.zones[id].mh_ambient;
            if (!this.mh_as_reference || this.zones[id].mh_address == 0) {

                /* if you want to use HVAC as reference or probe is missing */
                ambient = this.hvac.getAmbient(this.zones[id].zoneid);
            }
      
            var alarm = 0;
            if (this.zones[id].alarm != this.zones[id].bac_alarm) {

                alarm = this.zones[id].alarm;
            }

            this.zones[id].bac_alarm = this.zones[id].alarm;

            var bac_filter_alarm = 0;
            if (this.zones[id].filter != this.zones[id].bac_filter) {

                bac_filter_alarm = this.zones[id].filter;
            }

            this.zones[id].bac_filter = this.zones[id].filter;
      
            var frameout = "*#4*#%d#%d*51*%04d*%d*%d*%d*%s*%d*%s*%d*%d##".sprintf( this.zones[id].bac_area, this.zones[id].bac_address, this.zones[id].mh_setpoint * 10, this.zones[id].power, this._bkmodes.indexOf(this.zones[id].mode), this.zones[id].mh_fanspeed, (this.zones[id].mh_direction >= 0) ? "%d".sprintf( this.zones[id].mh_direction ) : "", ((bac_filter_alarm) ? 1 : 0), (ambient > 0) ? "%04.0f".sprintf(ambient*10) : "", ((alarm == 0) ? 0 : 1), alarm );
            if (this.zones[id].mh_setpoint != -1 && ambient != -1 && this.zones[id].mode != -1) {
       
                logger.debug(format("%s_MHH[u:%s] sendBacnetUpdate <%s>".sprintf(this.prefix,this.zones[id].hvac_address, frameout)));
                this.bacnetserver.sendFrame(frameout);

            } else {

                logger.debug(format("%s_MHH[u:%s] sendBacnetUpdate skipped, insufficient data <%s>".sprintf(this.prefix,this.zones[id].hvac_address,frameout)));
            }
      
        }
    
        this.setDriver = function (object) {

            this.hvac = object;
            this.name = this.hvac.name;
            this.prefix = this.hvac.prefix;
            this.hvac.onAmbientCallback = this.onAmbientCallback.bind(this);
            logger.info(format("%s_MHH[---] Adapter '%s' enabled".sprintf(this.prefix,this.name)));
        }
    
        this.setIPAddress = function (ipaddress) {

            this.hvac.setIPAddress(ipaddress);
        }

        this.setPort = function (port) {

            this.hvac.setPort(port);
        }
    
        /* Configure (and propagate) debug level */
        this.setDebug = function (_debug) {

            this.debug = _debug;
            this.hvac.setDebug(_debug);

            logger.debug("%s_MHH[---] Debug on".sprintf(this.prefix));
        }

        this.getDebug = function () {

            return this.debug;
        }

        this.getZoneByAddress = function (ownserver, address) {

            for (var zoneArrayIndex in this.zones) {
                if (this.zones[zoneArrayIndex].mh_ownserver == ownserver && this.zones[zoneArrayIndex].mh_address == address) {
                    return zoneArrayIndex;
                }
            }
            return -1;
        }

        // return zone array index from My Home zone address ( probe address ) set by user
        this.getZoneArrayIndexByMyHomeZoneAddress = function (ownserver, address) {

            for (var zoneArrayIndex in this.zones) {

                if (this.zones[zoneArrayIndex].mh_ownserver == ownserver && this.zones[zoneArrayIndex].mh_address == address) {

                    return zoneArrayIndex;
                }
            }
            return -1;
        }
    
        /* Find the zone index by bacnet address (area+address) */
        this.getZoneByBacnet = function(area,address) {
            for (var id in this.zones) {
                if (this.zones[id].bac_area == area && this.zones[id].bac_address == address) {
                    return id;
                }
            }
            return -1;      
        }
    
        /* Find the zone index by zoneid identifier, used by this and child driver to identify a zone */
        this.getZoneByZoneId = function(i) {
            for (var zoneArrayIndex in this.zones) {
                if (this.zones[zoneArrayIndex].zoneid == i) {
                    return zoneArrayIndex;
                }
            }
            return -1;      
        }    
    
        this.setMode = function(zoneArrayIndex, mode, submode) {

            var modeDescription;
            var subModeDescription;
            var previousMode;
            var previousSubMode;
            var previousModeDescription;
            var previousSubModeDescription;

            if (zoneArrayIndex == -1) return;

            previousMode = this.zones[zoneArrayIndex].mode;
            previousSubMode = this.zones[zoneArrayIndex].submode;

            if (mode > -1) {
                modeDescription = HVAC_Driver.modesDescription[mode];
            } else {
                modeDescription = "Unknown";
            }

            if (submode > -1) {
                subModeDescription = HVAC_Driver.modesDescription[submode];
            } else {
                subModeDescription = "Unknown";
            }

            logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] setMode, zone array index: %d, mode: %s (%d), submode: %s (%d)".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, zoneArrayIndex, modeDescription, mode, subModeDescription, submode));

            this.zones[zoneArrayIndex].submode = submode;

            switch (parseInt(mode)) {

                case mhOWN.OWN_THERMO_OFF:  // 0
                    this.zones[zoneArrayIndex].power = false;
                    this.zones[zoneArrayIndex].mode = mhOWN.OWN_THERMO_OFF;
                    //if (this.zones[zoneArrayIndex].mode == -1) this.zones[zoneArrayIndex].mode = HVAC_Driver.OWN_THERMO_AUTO;
                    //if (this.zones[zoneArrayIndex].mh_direction != -1) this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                    this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                    break;
        
                case mhOWN.OWN_THERMO_HEATING:  // 1
                    this.zones[zoneArrayIndex].power = true;
                    this.zones[zoneArrayIndex].mode = mhOWN.OWN_THERMO_HEATING;
                    if ((this.zones[zoneArrayIndex].bac_area == 0 && this.zones[zoneArrayIndex].bac_address == 0) || this.zones[zoneArrayIndex].heat_airdirection != -1) { /* If no TS is configured, 'force' direction */
                        this.zones[zoneArrayIndex].mh_direction = this.zones[zoneArrayIndex].heat_airdirection;
                        this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                    }
                    //if (this.zones[zoneArrayIndex].mh_direction != -1) this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                    break;
        
                case mhOWN.OWN_THERMO_COOLING:  // 2
                    this.zones[zoneArrayIndex].power = true;
                    this.zones[zoneArrayIndex].mode = mhOWN.OWN_THERMO_COOLING;
                    if ((this.zones[zoneArrayIndex].bac_area == 0 && this.zones[zoneArrayIndex].bac_address == 0) || this.zones[zoneArrayIndex].heat_airdirection != -1) { /* If no TS is configured, 'force' direction */
                        this.zones[zoneArrayIndex].mh_direction = this.zones[zoneArrayIndex].cool_airdirection;
                        this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                    }
                    //if (this.zones[zoneArrayIndex].mh_direction != -1) this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);          
                    break;
        
                case HVAC_Driver.OWN_THERMO_DRY:    // 101
                    this.zones[zoneArrayIndex].power = true;
                    this.zones[zoneArrayIndex].mode = HVAC_Driver.OWN_THERMO_DRY;
                    break;
            
                case HVAC_Driver.OWN_THERMO_FAN:    // 103
                    this.zones[zoneArrayIndex].power = true;
                    this.zones[zoneArrayIndex].mode = HVAC_Driver.OWN_THERMO_FAN;
                    break;
            
                case HVAC_Driver.OWN_THERMO_AUTO:   // 102
                    this.zones[zoneArrayIndex].power = true;
                    this.zones[zoneArrayIndex].mode = HVAC_Driver.OWN_THERMO_AUTO;
                    break;
            
                default:
                    this.zones[zoneArrayIndex].power = false;
                    break;
            }

            if (submode == mhOWN.OWN_THERMO_PROTECTION && this.off_thermal_protection) {
                this.zones[zoneArrayIndex].power = false;
        
                if (this.zones[zoneArrayIndex].mode == -1) this.zones[zoneArrayIndex].mode = HVAC_Driver.OWN_THERMO_AUTO;
                //if (this.zones[zoneArrayIndex].mh_direction != -1) this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
                logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] setMode, power OFF due to submode: PROTECTION and 'Protection Mode' preference set to OFF".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address));
            }
      
            //logger.debug("%s_MHH[u:%s] setMode power: %d -> %d, mode: %d -> %d".sprintf( this.prefix, this.zones[zoneArrayIndex].hvac_address, actualPower, this.zones[zoneArrayIndex].power, actualMode, this.zones[zoneArrayIndex].mode ) );

            if (previousMode > -1) {
                previousModeDescription = HVAC_Driver.modesDescription[previousMode];
            } else {
                previousModeDescription = "Unknown";
            }

            if (previousSubMode > -1) {
                previousSubModeDescription = HVAC_Driver.modesDescription[previousSubMode];
            } else {
                previousSubModeDescription = "Unknown";
            }

            if (mode > -1) {
                modeDescription = HVAC_Driver.modesDescription[mode];
            } else {
                modeDescription = "Unknown";
            }

            if (submode > -1) {
                subModeDescription = HVAC_Driver.modesDescription[submode];
            } else {
                subModeDescription = "Unknown";
            }

            logger.info("%s_MHH[gw:%02d,z:%02d,u:%s] setMode, mode: %s (%d) -> %s (%d), submode: %s (%d) -> %s (%d), power: %d".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address, previousModeDescription, previousMode, modeDescription, this.zones[zoneArrayIndex].mode, previousSubModeDescription, previousSubMode, subModeDescription, this.zones[zoneArrayIndex].submode, this.zones[zoneArrayIndex].power));
            
            // USELESS before a setLock? // this.hvac.setOperationMode(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mode);
            this.setLock(zoneArrayIndex,this.zones[zoneArrayIndex].lock);
            this.sendBacnetUpdate(zoneArrayIndex);
        }
    
        this.setLock = function (zoneArrayIndex, lock) {

            if (zoneArrayIndex == -1) return;

            var logHeading = "%s_MHH[gw:%02d,z:%02d,u:%s]".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address);
      
            if (this.zones[zoneArrayIndex].lock != lock) {
                logger.info("%s setLock new lock received: %s -> %s".sprintf(logHeading, HVAC_Driver.s_locks[this.zones[zoneArrayIndex].lock], HVAC_Driver.s_locks[lock]));
            } else {
                logger.info("%s setLock lock: %s".sprintf(logHeading, HVAC_Driver.s_locks[this.zones[zoneArrayIndex].lock]));
            }
      
            this.zones[zoneArrayIndex].lock = lock;
      
            // Resend whole information about mode and power state
            // Reports power state based on lock state and actuator selection
            var power = this.zones[zoneArrayIndex].power && (this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO);
            var mode = this.zones[zoneArrayIndex].mode;
      
            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_mode == HVAC_Driver.OWN_THERMO_NONE) power = false;
            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_mode == HVAC_Driver.OWN_THERMO_NONE) power = false;

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_mode == HVAC_Driver.OWN_THERMO_FANONLY) mode = HVAC_Driver.OWN_THERMO_FAN;
            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_mode == HVAC_Driver.OWN_THERMO_FANONLY) mode = HVAC_Driver.OWN_THERMO_FAN;
      
            // Consider setpoint reached state for fan speed adjust?
            //this.setpointReached(zoneArrayIndex) == 1 -> reached set point
            //this.setpointReached(zoneArrayIndex) == -2 -> impossible to determine if setpoint has been reached (ambient temperature or set point not available)
            //if option "When setpoint is reached" has been setted on "Turn OFF the unit"
            //if lock has not been setted
            //then hvac unit turn it off 
            //else turn it on until setpoint change
            var setPointReachedValue = this.setpointReached(zoneArrayIndex);

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING &&
                this.when_reached_heat == HVAC_Driver.REACHED_OFF &&
                this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO &&
                (setPointReachedValue == 1 || setPointReachedValue == -2)) {

                power = false;
                logger.info("%s setLock, power set to %s due to: set point reached, operation mode: HEATING and 'Heating when set point is reached' preference set to '%s'".sprintf(logHeading, power ? 'ON' : 'OFF', HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(this.when_reached_heat)]));
            }

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING &&
               this.when_reached_heat != HVAC_Driver.REACHED_OFF &&
               this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO &&
               (setPointReachedValue == 1 || setPointReachedValue == -2)) {

                logger.info("%s setLock, power set to %s due to: set point reached, operation mode: HEATING and 'Heating when set point is reached' preference set to '%s'".sprintf(logHeading, power ? 'ON' : 'OFF', HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(this.when_reached_heat)]));
            }

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING &&
                this.when_reached_cool == HVAC_Driver.REACHED_OFF &&
                this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO &&
                (setPointReachedValue == 1 || setPointReachedValue == -2)) {

                power = false;
                logger.info("%s setLock, power set to %s due to: set point reached, operation mode: COOLING and 'Cooling when set point is reached' preference set to '%s'".sprintf(logHeading, power ? 'ON' : 'OFF', HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(this.when_reached_cool)]));
            }

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING &&
               this.when_reached_cool != HVAC_Driver.REACHED_OFF &&
               this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO &&
               (setPointReachedValue == 1 || setPointReachedValue == -2)) {

                logger.info("%s setLock, power set to %s due to: set point reached, operation mode: COOLING and 'Cooling when set point is reached' preference set to '%s'".sprintf(logHeading, power ? 'ON' : 'OFF', HVAC_Driver.REACHED_VALUES_DESCRIPTION[HVAC_Driver.REACHED_VALUES_INDEXES.indexOf(this.when_reached_cool)]));
            }
      
            this.hvac.setPower(this.zones[zoneArrayIndex].zoneid, power);
            this.hvac.setOperationMode(this.zones[zoneArrayIndex].zoneid, mode);
                
            //+FEAT. Er2 Management
            // original -> this.updateLed(zoneArrayIndex, false);
            this.ledUpdate( zoneArrayIndex, forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] );
            //-FEAT. Er2 Management
            this.sharedTimer.start();
        }
    
        /* Alias */
        this.setSetPoint = function (zoneArrayIndex, temperature) {

            this.setSetpoint(zoneArrayIndex,temperature);
        }
    
        this.setSetpoint = function (zoneArrayIndex, temperature) {

            if( zoneArrayIndex == -1 ) return;

            var logHeading = "%s_MHH[gw:%02d,z:%02d,u:%s]".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address);

            logger.info("%s setSetpoint received setpoint: %.1f".sprintf(logHeading, temperature));
      
            // If setpoint changes and lock is "TILL SETPOINT CHANGE"
            if( this.zones[zoneArrayIndex].mh_setpoint != temperature && this.zones[zoneArrayIndex].lock == this.DRV_LOCK_TILL_SETPOINTCHANGE ) {
				
                logger.info("%s setSetpoint set point has changed, removing temporary lock!".sprintf(logHeading));
                this.setLock( zoneArrayIndex, this.DRV_LOCK_NO );
            }
      
            this.zones[zoneArrayIndex].mh_setpoint = temperature;
      
            /* HVAC Setpoint should be different? Useful when standard probes (and standard modes) are in use */
            var setpoint = temperature;

            if( this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_mode == HVAC_Driver.OWN_THERMO_INTEGRATE ) {
			
                var originalSetPoint = setpoint;
                setpoint = setpoint - this.heat_integrate_delta;

                logger.info("%s setSetpoint heat integration delta: %.1f, set point variation after integration correction: %.1f -> %.1f".sprintf(logHeading, this.heat_integrate_delta, originalSetPoint, setpoint));
            }
      
            if( this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_mode == HVAC_Driver.OWN_THERMO_INTEGRATE ) {
			
                var originalSetPoint = setpoint;
                setpoint = setpoint + this.cool_integrate_delta;

                logger.info("%s setSetpoint cool integration delta: %.1f, set point variation after integration correction: %.1f -> %.1f".sprintf(logHeading, this.cool_integrate_delta, originalSetPoint, setpoint));
            }
      
            this.zones[zoneArrayIndex].mh_setpoint_integrated = setpoint;

            logger.info("%s setSetpoint mode: %s (%d), heat adjust: %s, cool adjust: %s".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode, this.zones[zoneArrayIndex].heat_adjust, this.zones[zoneArrayIndex].cool_adjust));
      
            /* 2015.04.13 - Adjust setpoint, only if setpoint is not reached and hw ambient is valid */
            if( this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_adjust ) {

                var ambient = this.hvac.getAmbient(this.zones[zoneArrayIndex].zoneid);
                var compensationFlag = false;

                logger.info("%s setSetpoint hvac ambient temperature: %.1f, my home ambient temperature: %.1f, my home set point: %.1f".sprintf(logHeading, ambient, this.zones[zoneArrayIndex].mh_ambient, this.zones[zoneArrayIndex].mh_setpoint));

                if (!isNaN(ambient) && ambient > 0) {

                    if (this.zones[zoneArrayIndex].mh_setpoint > this.zones[zoneArrayIndex].mh_ambient) {

                        var originalSetPoint = setpoint;

                        if (this.zones[zoneArrayIndex].mh_ambient < ambient) {

                            var deltaCompensation = ambient - this.zones[zoneArrayIndex].mh_ambient;
                            setpoint = setpoint + deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f not reached, HVAC ambient temperature: %.1f is valid and higher than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, heat compensation delta: +%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (this.zones[zoneArrayIndex].mh_ambient > ambient) {

                            var deltaCompensation = this.zones[zoneArrayIndex].mh_ambient - ambient;
                            setpoint = setpoint - deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f not reached, HVAC ambient temperature: %.1f is valid and lower than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, heat compensation delta: -%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (compensationFlag === false) {

                            logger.info("%s setSetpoint no setpoint compensation due to my home ambient temperature: %.1f equal to hvac ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_ambient, ambient));
                        }

                    } else {

                        // mh_ambient >= mh_setpoint
                        // my home ambient temperature greater than or equal to setpoint. FORCED COMPENSATION
                        var originalSetPoint = setpoint;

                        if (this.zones[zoneArrayIndex].mh_ambient < ambient) {

                            var deltaCompensation = ambient - this.zones[zoneArrayIndex].mh_ambient;
                            setpoint = setpoint + deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f, HVAC ambient temperature: %.1f is valid and higher than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, forced cool compensation delta: +%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (this.zones[zoneArrayIndex].mh_ambient > ambient) {

                            var deltaCompensation = this.zones[zoneArrayIndex].mh_ambient - ambient;
                            setpoint = setpoint - deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f, HVAC ambient temperature: %.1f is valid and higher than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, forced heat compensation delta: -%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (compensationFlag === false) {

                            logger.info("%s setSetpoint no setpoint compensation due to my home ambient temperature: %.1f equal to hvac ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_ambient, ambient));
                        }
                    }

                } else {

                    // invalid or not positive ambient temperature value
                    if (isNaN(ambient)) {

                        logger.warning("%s setSetpoint no setpoint compensation due to invalid hvac ambient temperature".sprintf(logHeading));

                    } else {

                        logger.warning("%s setSetpoint no setpoint compensation due to not positive value hvac ambient temperature: %.1f".sprintf(logHeading, ambient));
                    }
                }
            }
      
            if(this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_adjust ) {

                var ambient = this.hvac.getAmbient(this.zones[zoneArrayIndex].zoneid);
                var compensationFlag = false;

                logger.info("%s setSetpoint hvac ambient temperature: %.1f, my home ambient temperature: %.1f, my home set point: %.1f".sprintf(logHeading, ambient, this.zones[zoneArrayIndex].mh_ambient, this.zones[zoneArrayIndex].mh_setpoint));

                if (!isNaN(ambient) && ambient > 0) {

                    if (this.zones[zoneArrayIndex].mh_setpoint < this.zones[zoneArrayIndex].mh_ambient) {

                        var originalSetPoint = setpoint;

                        if (this.zones[zoneArrayIndex].mh_ambient > ambient) {

                            var deltaCompensation = this.zones[zoneArrayIndex].mh_ambient - ambient;
                            setpoint = setpoint - deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f not reached, HVAC ambient temperature: %.1f is valid and lower than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, cool compensation delta: -%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (this.zones[zoneArrayIndex].mh_ambient < ambient) {

                            var deltaCompensation = ambient - this.zones[zoneArrayIndex].mh_ambient;
                            setpoint = setpoint + deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f not reached, HVAC ambient temperature: %.1f is valid and higher than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, heat compensation delta: +%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (compensationFlag === false) {

                            logger.info("%s setSetpoint no setpoint compensation due to my home ambient temperature: %.1f equal to hvac ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_ambient, ambient));
                        }

                    } else {

                        // mh_ambient >= mh_setpoint
                        // my home ambient temperature greater than or equal to setpoint. FORCED COMPENSATION
                        var originalSetPoint = setpoint;

                        if (this.zones[zoneArrayIndex].mh_ambient > ambient) {

                            var deltaCompensation = this.zones[zoneArrayIndex].mh_ambient - ambient;
                            setpoint = setpoint - deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f, HVAC ambient temperature: %.1f is valid and lower than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, forced cool compensation delta: -%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (this.zones[zoneArrayIndex].mh_ambient < ambient) {

                            var deltaCompensation = ambient - this.zones[zoneArrayIndex].mh_ambient;
                            setpoint = setpoint + deltaCompensation;

                            compensationFlag = true;

                            logger.warning("%s setSetpoint, set point: %.1f, HVAC ambient temperature: %.1f is valid and higher than My Home ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint, ambient, this.zones[zoneArrayIndex].mh_ambient));
                            logger.warning("%s setSetpoint, forced heat compensation delta: +%.1f, set point variation after compensation correction: %.1f -> %.1f".sprintf(logHeading, deltaCompensation, originalSetPoint, setpoint));
                        }

                        if (compensationFlag === false) {

                            logger.info("%s setSetpoint no setpoint compensation due to my home ambient temperature: %.1f equal to hvac ambient temperature: %.1f".sprintf(logHeading, this.zones[zoneArrayIndex].mh_ambient, ambient));
                        }
                    }

                } else {

                    // invalid or not positive ambient temperature value
                    if (isNaN(ambient)) {

                        logger.warning("%s setSetpoint no setpoint compensation due to invalid hvac ambient temperature".sprintf(logHeading));

                    } else {

                        logger.warning("%s setSetpoint no setpoint compensation due to not positive value hvac ambient temperature: %.1f".sprintf(logHeading, ambient));
                    }
                }
            }
      
            logger.info("%s setSetpoint, received set point: %.1f, configured set point: %.1f".sprintf(logHeading, temperature, setpoint));

            var setPointReachedValue = this.setpointReached(zoneArrayIndex);
            logger.info("%s setSetpoint set point reached: %s".sprintf(logHeading, (setPointReachedValue == 1)));

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.when_reached_heat == HVAC_Driver.REACHED_FAN_LOW && this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO && setPointReachedValue == 1) {

                setpoint = Math.round(setpoint - 0.5);

            } else if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.when_reached_cool == HVAC_Driver.REACHED_FAN_LOW && this.zones[zoneArrayIndex].lock == this.DRV_LOCK_NO && setPointReachedValue == 1) {

                setpoint = Math.round(setpoint + 0.5);

            } else {

                switch (this.zones[zoneArrayIndex].mode) {

                    case mhOWN.OWN_THERMO_HEATING:

                        /* Ceil setpoint on heating */
                        setpoint = Math.ceil(setpoint);
                        break;

                    case mhOWN.OWN_THERMO_COOLING:

                        /* Floor setpoint on cooling */
                        setpoint = Math.floor(setpoint);
                        break;

                    default:

                        setpoint = Math.round(setpoint);
                        break;
                }
            }

            logger.info("%s setSetpoint rounded set point: %.1f".sprintf(logHeading, setpoint));

            this.hvac.setSetpoint( this.zones[zoneArrayIndex].zoneid, setpoint );
      
            this.sendBacnetUpdate( zoneArrayIndex );		

            this.ledUpdate( zoneArrayIndex, forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] );
        }
    
        this.setAmbient = function (zoneArrayIndex, temperature) {

            if (zoneArrayIndex == -1) return;
      
            this.zones[zoneArrayIndex].mh_ambient = temperature;
            logger.info("%s_MHH[u:%s] setAmbient <%.1f>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,temperature));
      
            this.hvac.setAmbient(this.zones[zoneArrayIndex].zoneid,temperature);
            this.sendBacnetUpdate(zoneArrayIndex);
            //this.updateLed(zoneArrayIndex,false);
            //+FEAT. Er2 Management
            this.ledUpdate( zoneArrayIndex, forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] );
            //-FEAT. Er2 Management
      
            if (this.mh_as_reference) this.forces(zoneArrayIndex);
        }    

        this.forces = function (zoneArrayIndex) {

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING) {

                switch (this.when_reached_heat) {
                    case HVAC_Driver.REACHED_NONE:
                        break;

                    case HVAC_Driver.REACHED_FAN_LOW:
                    case HVAC_Driver.REACHED_FAN_AUTO:
                        this.setFanSpeed(zoneArrayIndex, this.zones[zoneArrayIndex].mh_fanspeed); /* Recommit fanspeed */
                        break;

                    case HVAC_Driver.REACHED_OFF:
                        this.setLock(zoneArrayIndex, this.zones[zoneArrayIndex].lock); /* Recommit power */
                        break;
                }
            }

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) {

                switch (this.when_reached_cool) {
                    case HVAC_Driver.REACHED_NONE:
                        break;

                    case HVAC_Driver.REACHED_FAN_LOW:
                    case HVAC_Driver.REACHED_FAN_AUTO:
                        this.setFanSpeed(zoneArrayIndex, this.zones[zoneArrayIndex].mh_fanspeed); /* Recommit fanspeed */
                        break;

                    case HVAC_Driver.REACHED_OFF:
                        this.setLock(zoneArrayIndex, this.zones[zoneArrayIndex].lock); /* Recommit power */
                        break;
                }
            }
        }
    
        this.setFanSpeed = function (zoneArrayIndex, speed) {

            if( zoneArrayIndex == -1 ) return;
      
            this.zones[zoneArrayIndex].mh_fanspeed = speed;
            logger.info( "%s_MHH[u:%s] setFanSpeed <%d>".sprintf( this.prefix, this.zones[zoneArrayIndex].hvac_address, speed ) );
      
            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING) {

                if (this.when_reached_heat == HVAC_Driver.REACHED_FAN_LOW || this.when_reached_heat == HVAC_Driver.REACHED_FAN_AUTO) {

                    if (this.setpointReached(zoneArrayIndex) == 1) {

                        speed = this.when_reached_heat;
                    }
                }
            }

            if (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) {

                if (this.when_reached_cool == HVAC_Driver.REACHED_FAN_LOW || this.when_reached_cool == HVAC_Driver.REACHED_FAN_AUTO) {

                    if (this.setpointReached(zoneArrayIndex) == 1) {

                        speed = this.when_reached_cool;
                    }
                }
            }
      
            if (this.zones[zoneArrayIndex].autofan || speed != mhOWN.OWN_THERMO_FANSPEED_AUTO) {

                this.hvac.setFanSpeed(this.zones[zoneArrayIndex].zoneid, speed);

            } else {

                /* Calc fan speed based on mh or hvac ambient temperature */
                var ambient = this.zones[zoneArrayIndex].mh_ambient;
                var setpoint = this.zones[zoneArrayIndex].mh_setpoint_integrated;

                if ((!this.mh_as_reference) || this.zones[zoneArrayIndex].mh_address == 0) {
                    /* if you want to use HVAC as reference or probe is missing */
                    ambient = this.hvac.getAmbient( this.zones[zoneArrayIndex].zoneid );
                    setpoint = this.hvac.getSetpoint( this.zones[zoneArrayIndex].zoneid );
                }

                this.hvac.setFanSpeed( this.zones[zoneArrayIndex].zoneid, this.calcFanSpeed( zoneArrayIndex, ambient, setpoint, this.zones[zoneArrayIndex].mode ) );        
            }
        }
    
        /* Returns 1 (reached), 0 (not reached) or -1 (not managed) */
        this.setpointReached_old = function(zoneArrayIndex) {
      
            /* Retreive reference ambient/setpoint temperature ... */
            var ambient = this.zones[zoneArrayIndex].mh_ambient;
            var setpoint = this.zones[zoneArrayIndex].mh_setpoint_integrated;
      
            if ((!this.mh_as_reference) || this.zones[zoneArrayIndex].mh_address == 0) { /* if you want to use HVAC as reference or probe is missing */
                ambient = this.hvac.getAmbient(this.zones[zoneArrayIndex].zoneid);
                setpoint = this.hvac.getSetpoint(this.zones[zoneArrayIndex].zoneid);
            }      
      
            logger.debug("%s_MHH[u:%s] setpointReached <ambient=%.1f,setpoint=%.1f,mode=%d>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,ambient,setpoint,this.zones[zoneArrayIndex].mode));
            if (ambient == -1 || setpoint == -1) return -2;
      
            switch (this.zones[zoneArrayIndex].mode) {
                case mhOWN.OWN_THERMO_HEATING:
                    if (ambient >= setpoint) {
                        return 1;
                    }
                    return 0;
                    break;
                case mhOWN.OWN_THERMO_COOLING:
                    if (ambient <= setpoint) {
                        return 1;
                    }
                    return 0;
                    break;
        
            }
            return -1;
        }

        this.setpointReached = function (zoneArrayIndex) {

            /* Retrieve reference ambient/setpoint temperature ... */
            var ambient = this.zones[zoneArrayIndex].mh_ambient;
            var setpoint = this.zones[zoneArrayIndex].mh_setpoint_integrated;
            var previousSetPointReached;

            var logHeading = "%s_MHH[gw:%02d,z:%02d,u:%s] setpointReached".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address);

            if ((!this.mh_as_reference) || this.zones[zoneArrayIndex].mh_address == 0) { /* if you want to use HVAC as reference or probe is missing */
                ambient = this.hvac.getAmbient(this.zones[zoneArrayIndex].zoneid);
                setpoint = this.hvac.getSetpoint(this.zones[zoneArrayIndex].zoneid);
                logger.warning("%s setting myhome ambient temperature and myhome setpoint to hvac system values: %.1f, %.1f due to hvac system acting as reference".sprintf(logHeading, ambient, setpoint));
            }

            if (ambient == -1 || setpoint == -1) {
                logger.warning("%s unable to state if set point has been reached due to invalid ambient: %.1f or invalid setpoint: %.1f, mode: %d".sprintf(logHeading, ambient, setpoint, this.zones[zoneArrayIndex].mode));
                return -2;
            }

            switch (this.zones[zoneArrayIndex].mode) {

                case mhOWN.OWN_THERMO_HEATING:

                    if (this.zones[zoneArrayIndex].hysteresis == true) {
                        var prevSetPoint = setpoint;
                        setpoint -= this.heat_hysteresis;
                        logger.debug("%s subtracting hysteresis value, new setpoint value: (setpoint: %.1f - heat hysteresis: %.1f) : %.1f".sprintf(logHeading, prevSetPoint, this.heat_hysteresis, setpoint));
                    }

                    if (ambient >= setpoint) {

                        previousSetPointReached = this.zones[zoneArrayIndex].setPointReached;
                        this.zones[zoneArrayIndex].setPointReached = true;
                        logger.debug("%s previousSetPointReached: %s, setpointReached: true".sprintf(logHeading, previousSetPointReached ? "true" : "false"));

                        // transition set point not reached -> set point reached, subtract hysteresis delta to set point
                        if (this.zones[zoneArrayIndex].setPointReached != previousSetPointReached) {
                            this.zones[zoneArrayIndex].hysteresis = true;
                            logger.debug("%s previousSetPointReached: %s different from setpointReached: true so set hysteresis flag: true".sprintf(logHeading, previousSetPointReached ? "true" : "false"));
                        }

                        logger.debug("%s setpoint reached: TRUE, ambient: %.1f, setpoint: %.1f, mode: %s (%d)".sprintf(logHeading, ambient, setpoint, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode));
                        return 1;

                    } else {

                        previousSetPointReached = this.zones[zoneArrayIndex].setPointReached;
                        this.zones[zoneArrayIndex].setPointReached = false;
                        logger.debug("%s previousSetPointReached: %s, setpointReached: false".sprintf(logHeading, previousSetPointReached ? "true" : "false"));

                        // transition set point reached -> set point not reached, set setpoint without hysteresis delta
                        if (this.zones[zoneArrayIndex].setPointReached != previousSetPointReached) {
                            this.zones[zoneArrayIndex].hysteresis = false;
                            logger.debug("%s previousSetPointReached: %s different from setpointReached: false so set hysteresis falg false".sprintf(logHeading, previousSetPointReached ? "true" : "false"));
                        }

                        logger.info("%s setpoint reached: FALSE, ambient: %.1f, setpoint: %.1f, mode: %s (%d)".sprintf(logHeading, ambient, setpoint, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode));
                        return 0;
                    }
                    break;

                case mhOWN.OWN_THERMO_COOLING:

                    if (this.zones[zoneArrayIndex].hysteresis == true) {
                        var prevSetPoint = setpoint;
                        setpoint += this.cool_hysteresis;
                        logger.debug("%s adding hysteresis value, new setpoint value: (setpoint: %.1f + cool hysteresis: %.1f) : %.1f".sprintf(logHeading, prevSetPoint, this.cool_hysteresis, setpoint));
                    }

                    if (ambient <= setpoint) {

                        previousSetPointReached = this.zones[zoneArrayIndex].setPointReached;
                        this.zones[zoneArrayIndex].setPointReached = true;
                        logger.debug("%s previousSetPointReached: %s, setpointReached: true".sprintf(logHeading, previousSetPointReached ? "true" : "false"));

                        // transition set point not reached -> set point reached, add hysteresis delta to set point
                        if (this.zones[zoneArrayIndex].setPointReached != previousSetPointReached) {
                            this.zones[zoneArrayIndex].hysteresis = true;
                            logger.debug("%s previousSetPointReached: %s different from setpointReached: true so set hysteresis flag: true".sprintf(logHeading, previousSetPointReached ? "true" : "false"));
                        }

                        logger.debug("%s set point reached: TRUE, ambient: %.1f, setpoint: %.1f, mode: %s (%d)".sprintf(logHeading, ambient, setpoint, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode));
                        return 1;

                    } else {

                        previousSetPointReached = this.zones[zoneArrayIndex].setPointReached;
                        this.zones[zoneArrayIndex].setPointReached = false;
                        logger.debug("%s previousSetPointReached: %s, setpointReached: false".sprintf(logHeading, previousSetPointReached ? "true" : "false"));

                        // transition set point reached -> set point not reached, set setpoint without hysteresis delta
                        if (this.zones[zoneArrayIndex].setPointReached != previousSetPointReached) {
                            this.zones[zoneArrayIndex].hysteresis = false;
                            logger.debug("%s previousSetPointReached: %s different from setpointReached: false so set hysteresis flag: false".sprintf(logHeading, previousSetPointReached ? "true" : "false"));
                        }

                        logger.debug("%s set point reached: FALSE, ambient: %.1f, setpoint: %.1f, mode: %s (%d)".sprintf(logHeading, ambient, setpoint, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode));
                        return 0;
                    }
                    break;

            }

            logger.warning("%s unable to state if set point has been reached due to mode: %s (%d) other than heating or cooling, ambient: %.1f or invalid setpoint: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], this.zones[zoneArrayIndex].mode, ambient, setpoint));
            return -1;
        }
    
        this.setOrientation = function(zoneArrayIndex,orientation) {
            if (zoneArrayIndex == -1) return;
      
            this.zones[zoneArrayIndex].mh_direction = orientation;
            this.hvac.setOrientation(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].mh_direction);
            logger.debug("%s_MHH[u:%s] setOrientation <%d>".sprintf(this.prefix,this.zones[zoneArrayIndex].hvac_address,this.zones[zoneArrayIndex].mh_direction));
        }
    
        /* Update gateway led status, only if required */
        this.ledUpdate = function( zoneArrayIndex, bypass ) {

            var modeDescription;
            var subModeDescription;
            var fanSpeed = -1;
            var mode = -1;

            if (zoneArrayIndex == -1) {
                logger.warning("%s_MHH[---] LED UPDATE SKIPPED due to invalid zone array index = -1".sprintf(this.prefix));
                logger.debug("%s_MHH[---] -ledUpdate".sprintf(this.prefix));
                return;
            }

            var logHeading = "%s_MHH[gw:%02d,z:%02d,u:%s]".sprintf(this.prefix, this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, this.zones[zoneArrayIndex].hvac_address);
		
            var currentLedStatus = this.zones[zoneArrayIndex].led_status;

            if( this.zones[zoneArrayIndex].mode > -1 ) {
                modeDescription = HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode];
            } else {
                modeDescription = "Unknown";
            }

            if (this.zones[zoneArrayIndex].submode > -1) {
                subModeDescription = HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode];
            } else {
                subModeDescription = "Unknown";
            }

            logger.debug("%s +ledUpdate, zone array index: %d bypass: %s".sprintf(logHeading, zoneArrayIndex, bypass ? "true" : "false"));
            logger.debug("%s address: %d, mode: %s (%d), submode: %s (%d), power: %d, heat mode: %s, cool mode: %s".sprintf(logHeading, this.zones[zoneArrayIndex].mh_address, modeDescription, this.zones[zoneArrayIndex].mode, subModeDescription, this.zones[zoneArrayIndex].submode, this.zones[zoneArrayIndex].power, this.zones[zoneArrayIndex].heat_mode, this.zones[zoneArrayIndex].cool_mode));

            if (this.zones[zoneArrayIndex].mh_address == 0) {
                logger.warning("%s LED UPDATE SKIPPED due to invalid My Home zone address: 0".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            if (this.zones[zoneArrayIndex].mh_setpoint < 0) {
                logger.warning("%s LED UPDATE SKIPPED due to invalid My Home set point: %d".sprintf(logHeading, this.zones[zoneArrayIndex].mh_setpoint));
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            if (this.zones[zoneArrayIndex].mode < 0) {
                logger.warning("%s LED UPDATE SKIPPED due to unknown mode".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            //if bypass is true and zone does not exist skip control to send gateway update
            if(!bypass) {

                var hWExist = this.hvac.getHWExist(this.zones[zoneArrayIndex].zoneid);

                logger.debug("%s ledUpdate zones[%d].zoneid: %d, hWExist: %s".sprintf(logHeading, zoneArrayIndex, this.zones[zoneArrayIndex].zoneid, hWExist ? "true" : "false"));

                if( !hWExist )
                {
                    logger.info("%s LED UPDATE SKIPPED due to hvac zone does not exist".sprintf(logHeading));
                    logger.debug("%s -ledUpdate".sprintf(logHeading));
                    return;
                }
            } else {
                logger.debug("%s ledUpdate skipping hvac zone existence check".sprintf(logHeading));
            }

            /* Continue only if needed */
            // original -> if (!(force || (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_actuators == 'VIRT') || (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_actuators == 'VIRT')))
            // if status led update is forced then go on
            // if mode is OFF then go on
            // if mode is HEATING and heat actuator is not PHYSICAL then go on
            // if mode is COOLING and cool actuator is not PHYSICAL then go on
            // otherwise skip status led update and exit
            // In other words:
            // 1) if mode is OFF then skip led update
            // 2) if mode is HEATING and heating actuator is PHYSICAL then skip led update
            // 3) if mode is COOLING and cooling actuator is PHYSICAL then skip led update
            // otherwise go on
            if (!(
                (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_OFF) ||
                (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_actuators == 'VIRT') ||
                (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_actuators == 'VIRT'))) {

                logger.debug("%s ledUpdate ( mode == mhOWN.OWN_THERMO_OFF ): %s".sprintf(logHeading, (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_OFF) ? "TRUE" : "FALSE"));
                logger.debug("%s ledUpdate ( mode == mhOWN.OWN_THERMO_HEATING && heat_actuators == 'VIRT' ): %s".sprintf(logHeading, (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_actuators == 'VIRT') ? "TRUE" : "FALSE"));
                logger.debug("%s ledUpdate ( mode == mhOWN.OWN_THERMO_COOLING && cool_actuators == 'VIRT' ): %s".sprintf(logHeading, (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_actuators == 'VIRT') ? "TRUE" : "FALSE"));
                logger.warning("%s LED UPDATE SKIPPED due to operation mode OFF or physical actuator set".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));

                return;
            }

            logger.info("%s ledUpdate current led status: %d, mode: %d, power: %d, heat mode: %s, cool mode: %s, bypass: %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].mode, this.zones[zoneArrayIndex].power, this.zones[zoneArrayIndex].heat_mode, this.zones[zoneArrayIndex].cool_mode, bypass));

            //mhOWN.ThermoZone_GatewayUpdate( <gateway id>, <my home zone address>, <fan speed>, <mode>, <set point> );

            // fan speed    hex      description
            // 0            00       Automatic
            // 1            01       Minimum fan speed
            // 2            02       Medium fan speed
            // 3            03       Maximum fan speed
            // 4            04       Silent mode
            // 13           0D       Fan-coil power on, heating
            // 14           0E       Fan-coil power on, cooling
            // 15           0F       Fan-coil power off, setpoint reached
				
            // mode     description
            // 0        Heating
            // 1        Cooling				
            // 2        Anti freeze protection
            // 3        Thermal protection
            // 4        Generic protection
            // 5        Fan
            // 6        Dry
            // 7        Automatic
            // 15       OFF

            // CHECK LOCK OR TEMPORARY LOCK IS ON
            if( this.zones[zoneArrayIndex].lock != this.DRV_LOCK_NO ) {

                // if the set point has not been reached the probe's flame or icicle icon must remain switched on
                // if the set point has been reached we have to check user preferences about lock behaviour
                // to switch on or off the probe's flame or icicle
                var reached = this.setpointReached(zoneArrayIndex);

                switch (reached) {

                    case -2:
                    case -1:
                        // -1 if mode is other than heating and cooling
                        // -2 if ambient temperature value or set point value is -1 that is at system startup
                        this.zones[zoneArrayIndex].led_status = 0;
                        break;

                    case 0: // not reached
                        this.zones[zoneArrayIndex].led_status = 1;
                        break;

                    case 1: // reached
                        //led status always follows the rule when set point is reached led status is OFF, otherwise led status is ON
                        this.zones[zoneArrayIndex].led_status = 0;
                        break;
                }

                if (this.zones[zoneArrayIndex].led_status == 1) {
                    //keep current fan speed value
                    fanSpeed = this.zones[zoneArrayIndex].mh_fanspeed;
                } else {
                    fanSpeed = 15;
                }

                switch (this.zones[zoneArrayIndex].mode) {

                    case mhOWN.OWN_THERMO_COOLING:  // 2

                        switch (this.zones[zoneArrayIndex].submode) {

                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                            case mhOWN.OWN_THERMO_AUTO:     // 11
                                mode = 1;       // Cooling
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14
                                // Thermal protection:
                                // mode: 3, description: Thermal protection
                                mode = 3;       // Thermal protection
                                break;

                            default:                            
                                mode = 1;       // Cooling
                        }
                        break;

                    case mhOWN.OWN_THERMO_HEATING:  // 1

                        switch (this.zones[zoneArrayIndex].submode) {
                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                            case mhOWN.OWN_THERMO_AUTO:     // 11                            
                                mode = 0;       // Heating
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14
                                // Anti freeze protection
                                // fan speed: 15, hex: 0F, description: fan-coil power off, setpoint reached
                                // mode:2, description: Anti freeze protection
                                mode = 2;       // Anti freeze protection
                                break;

                            default:
                                mode = 0;       // Heating
                        }
                        break;

                    case mhOWN.OWN_THERMO_OFF:  // 0
                        fanSpeed = 15;  // Off
                        mode = 15;      // Off
                        this.zones[zoneArrayIndex].led_status = 0;
                        break;
                }

                //this.hvac.setLedStatus(this.zones[zoneArrayIndex].zoneid, this.zones[zoneArrayIndex].led_status);
                logger.info("%s ledUpdate led status transition: %d -> %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].led_status));
                logger.info("%s ledUpdate ThermoZone_GatewayUpdate (%s, %s) lock on, ownserver: %d, zone: %d, fan speed: %s (%d), mode: %d, set point: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, HVAC_Driver.s_fanspeeds[fanSpeed], fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = false;

                // +Better Er2 management
                if (this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    logger.debug("%s startupLedStatusUpdate: TRUE -> FALSE".sprintf(logHeading));
                    this.zones[zoneArrayIndex].startupLedStatusUpdate = false;
                }

                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.stop();
                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.start();
                logger.debug("%s ledUpdate zone: %d gatewayStatusUpdateTimer restart".sprintf(logHeading, this.zones[zoneArrayIndex].zoneid));
                // -Better Er2 management

                logger.info("%s LED UPDATE!!! [0]".sprintf(logHeading));

                logger.debug("%s -ledUpdate".sprintf(this.prefix, this.zones[zoneArrayIndex].hvac_address));
                return;
            }


            // Se sono in Riscaldamento e nella configurazione della zona, nella combo "Modalit riscaldamento" imposto l'opzione "Non usare questo sistema
            // oppure
            // se sono in Raffrescamento e nella configurazione della zona, nella combo "Modalit raffrescamento" imposto l'opzione "Non usare questo sistema
            // allora se il led di stato  acceso ( vale a dire se la macchina sta chiamando ) lo spengo e aggiorno il led di stato della sonda
            if ((this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_HEATING && this.zones[zoneArrayIndex].heat_mode == HVAC_Driver.OWN_THERMO_NONE) || 
                (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING && this.zones[zoneArrayIndex].cool_mode == HVAC_Driver.OWN_THERMO_NONE)) {

                if (this.zones[zoneArrayIndex].led_status != 0) {
                    this.zones[zoneArrayIndex].led_status = 0;

                    mode = (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) ? 1 : 0;
                    fanSpeed = 15;

                    //this.hvac.setLedStatus(this.zones[zoneArrayIndex].zoneid,this.zones[zoneArrayIndex].led_status);

                    logger.info("%s ledUpdate led status transition: %d -> %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].led_status));
                    logger.info("%s ledUpdate ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %s (%d), mode: %d, set point: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, HVAC_Driver.s_fanspeeds[fanSpeed], fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                    //mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver,this.zones[zoneArrayIndex].mh_address,15,((this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) ? 1:0),this.zones[zoneArrayIndex].mh_setpoint);
                    mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                    //+gestione Er2
                    forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = false;
                    //-gestione Er2

                    // +Better Er2 management
                    if(this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                        logger.debug("%s startupLedStatusUpdate: TRUE -> FALSE".sprintf(logHeading));
                        this.zones[zoneArrayIndex].startupLedStatusUpdate = false;
                    }

                    // when the status led is updated start the timer
                    this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.stop();
                    this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.start();
                    logger.debug("%s ledUpdate zone: %d gatewayStatusUpdateTimer restart".sprintf(logHeading, this.zones[zoneArrayIndex].zoneid));
                    // -Better Er2 management

                    logger.info("%s: LED UPDATE!!! [1]".sprintf(logHeading));
                } else {
                    logger.warning("%s LED UPDATE SKIPPED due to operation mode heating or cooling and system is not used".sprintf(logHeading));
                }
        
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            // CHECK POWER OFF
            if (!this.zones[zoneArrayIndex].power) {

                this.zones[zoneArrayIndex].led_status = 0;
                fanSpeed = 15;  // Off

                switch(this.zones[zoneArrayIndex].mode) {

                    case mhOWN.OWN_THERMO_COOLING:  // 2    		        

                        switch (this.zones[zoneArrayIndex].submode) {

                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                            case mhOWN.OWN_THERMO_AUTO:     // 11
                                mode = 1;       // Cooling
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14
                                // Thermal protection:
                                // mode: 3, description: Thermal protection
                                mode = 3;       // Thermal protection
                                break;

                            default:
                                mode = 1;       // Cooling
                        }
                        break;

                    case mhOWN.OWN_THERMO_HEATING:  // 1

                        switch(this.zones[zoneArrayIndex].submode) {

                            case mhOWN.OWN_THERMO_MANUAL:   // 10
                            case mhOWN.OWN_THERMO_AUTO:     // 11
                                mode = 0;       // Heating
                                break;

                            case mhOWN.OWN_THERMO_PROTECTION:   // 14
                                // Anti freeze protection
                                // mode:2, description: Anti freeze protection
                                mode = 2;       // Anti freeze protection
                                break;

                            default:
                                mode = 0;       // Heating
                        }
                        break;

                    case mhOWN.OWN_THERMO_OFF:  // 0    		        
                        mode = 15;      // Off
                        break;
                }

                logger.info("%s ledUpdate led status transition: %d -> %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].led_status));
                logger.info("%s ledUpdate ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %s (%d), mode: %d, set point: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, HVAC_Driver.s_fanspeeds[fanSpeed], fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                //+gestione Er2
                forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = false;
                //-gestione Er2

                // +Better Er2 management
                if(this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    logger.debug("%s startupLedStatusUpdate: TRUE -> FALSE".sprintf(logHeading));
                    this.zones[zoneArrayIndex].startupLedStatusUpdate = false;
                }

                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.stop();
                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.start();
                logger.debug("%s ledUpdate zone: %d gatewayStatusUpdateTimer restart".sprintf(logHeading, this.zones[zoneArrayIndex].zoneid));
                // -Better Er2 management

                logger.info("%s LED UPDATE!!! [2]".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            // CHECK SET POINT REACHED
            var reached = this.setpointReached(zoneArrayIndex);
            if (reached != 0) {
            
                this.zones[zoneArrayIndex].led_status = 0;
                mode = (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) ? 1 : 0;
                fanSpeed = 15;

                logger.info("%s ledUpdate led status transition: %d -> %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].led_status));
                logger.info("%s ledUpdate ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %s (%d), mode: %d, set point: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, HVAC_Driver.s_fanspeeds[fanSpeed], fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                //mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver,this.zones[zoneArrayIndex].mh_address,15,((this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) ? 1:0),this.zones[zoneArrayIndex].mh_setpoint);
                mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                //+gestione Er2
                forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = false;
                //-gestione Er2

                // +Better Er2 management
                if(this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    logger.debug("%s startupLedStatusUpdate: TRUE -> FALSE".sprintf(logHeading));
                    this.zones[zoneArrayIndex].startupLedStatusUpdate = false;
                }

                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.stop();
                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.start();
                logger.info("%s ledUpdate zone: %d gatewayStatusUpdateTimer restart".sprintf(logHeading, this.zones[zoneArrayIndex].zoneid));
                // -Better Er2 management

                logger.info("%s: LED UPDATE!!! [3]".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));
                return;
            }

            // CHECK LED STATUS OFF
            if (this.zones[zoneArrayIndex].led_status != 1) {
                this.zones[zoneArrayIndex].led_status = 1;

                mode = (this.zones[zoneArrayIndex].mode == mhOWN.OWN_THERMO_COOLING) ? 1 : 0;
                fanSpeed = this.zones[zoneArrayIndex].mh_fanspeed;

                logger.info("%s ledUpdate led status transition: %d -> %d".sprintf(logHeading, currentLedStatus, this.zones[zoneArrayIndex].led_status));
                logger.info("%s ledUpdate ThermoZone_GatewayUpdate (%s, %s) ownserver: %d, zone: %d, fan speed: %s (%d), mode: %d, set point: %.1f".sprintf(logHeading, HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].mode], HVAC_Driver.modesDescription[this.zones[zoneArrayIndex].submode], this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, HVAC_Driver.s_fanspeeds[fanSpeed], fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint));
                mhOWN.ThermoZone_GatewayUpdate(this.zones[zoneArrayIndex].mh_ownserver, this.zones[zoneArrayIndex].mh_address, fanSpeed, mode, this.zones[zoneArrayIndex].mh_setpoint);

                //+gestione Er2
                forcedLedUpdate[this.zones[zoneArrayIndex].zoneid] = false;
                //-gestione Er2

                // +Better Er2 management
                if(this.zones[zoneArrayIndex].startupLedStatusUpdate) {
                    logger.debug("%s startupLedStatusUpdate: TRUE -> FALSE".sprintf(logHeading));
                    this.zones[zoneArrayIndex].startupLedStatusUpdate = false;
                }

                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.stop();
                this.zones[zoneArrayIndex].gatewayStatusUpdateTimer.start();
                logger.info("%s ledUpdate zone: %d gatewayStatusUpdateTimer restart".sprintf(logHeading, this.zones[zoneArrayIndex].zoneid));
                // -Better Er2 management

                logger.info("%s LED UPDATE!!! [4]".sprintf(logHeading));
                logger.debug("%s -ledUpdate".sprintf(logHeading));

                return;
            }

            logger.info("%s LED UPDATE SKIPPED".sprintf(logHeading));
            logger.debug("%s -ledUpdate".sprintf(logHeading));
        }
    
        /* Calculate fan speed auto if is not supported by device. Return 1,2,3 speed */
        this.calcFanSpeed = function(id,ambient,setpoint,mode) {
            var fans = mhOWN.OWN_THERMO_FANSPEED_LOW; /* Safe level */
            logger.info("%s_MHH[u:%s] Auto fanspeed calc (%.1f,%.1f,%d)...".sprintf(this.prefix,this.zones[id].hvac_address,ambient,setpoint,mode));
            switch (mode) {
                /* TODO What can we do when operation mode is DRY and FAN or (worst case) AUTO? */
          
                case mhOWN.OWN_THERMO_HEATING:
                    if(ambient<(setpoint-((this.heat_fandelta/3)*2))) { fans = mhOWN.OWN_THERMO_FANSPEED_HIGH; }
                    if(ambient>=(setpoint-((this.heat_fandelta/3)*2))) { fans = mhOWN.OWN_THERMO_FANSPEED_MID; }
                    if(ambient>=(setpoint-(this.heat_fandelta/3))) { fans = mhOWN.OWN_THERMO_FANSPEED_LOW; }
            
                    logger.info("%s_MHH[u:%s] Auto fanspeed becomes <%s> (ambient=%.1f, delta=%.1f, setpoint=%.1f, mode=%s)".sprintf(
                      this.prefix, this.zones[id].hvac_address, HVAC_Driver.s_fanspeeds[fans], ambient, this.heat_fandelta, setpoint, HVAC_Driver.s_modes[mode]));
                    break;
            
                case mhOWN.OWN_THERMO_COOLING:
                    if(ambient>(setpoint+(this.cool_fandelta/3)*2)) { fans = mhOWN.OWN_THERMO_FANSPEED_HIGH; }
                    if(ambient<=(setpoint+((this.cool_fandelta/3)*2))) { fans = mhOWN.OWN_THERMO_FANSPEED_MID; }
                    if(ambient<=(setpoint+((this.cool_fandelta/3)))) { fans =  mhOWN.OWN_THERMO_FANSPEED_LOW;}
            
                    logger.info("%s_MHH[u:%s] Auto fanspeed becomes <%s> (ambient=%.1f, delta=%.1f, setpoint=%.1f, mode=%s)".sprintf(
                      this.prefix, this.zones[id].hvac_address, HVAC_Driver.s_fanspeeds[fans], ambient, this.cool_fandelta, setpoint, HVAC_Driver.s_modes[mode]));
                    break;
            }
            return fans;
        };
    
        /* Lock */
    
        /* Event, do not exit on first occurrence */
        this.cenEvent = function(ownserver,bus,a,pl,button) {
            for (var id in this.zones) {
                if (this.zones[id].mh_ownserver == ownserver && this.zones[id].lock_cen_bus == bus && this.zones[id].lock_cen_a == a && this.zones[id].lock_cen_pl == pl) {
                    if (this.zones[id].lock_button_lock == button) this.setLock(id,this.DRV_LOCK_YES); 
                    if (this.zones[id].lock_button_unlock == button) this.setLock(id,this.DRV_LOCK_NO);
                    if (this.zones[id].lock_button_temporary == button) this.setLock(id,this.DRV_LOCK_TILL_SETPOINTCHANGE);
                }
            }
        }
    
        /* Event, do not exit on first occurrence */
        this.cenplusEvent = function(ownserver,address,button) {
            for (var id in this.zones) {
                if (this.zones[id].mh_ownserver == ownserver && this.zones[id].lock_cenplus == address) {
                    if (this.zones[id].lock_button_lock == button) this.setLock(id,this.DRV_LOCK_YES); 
                    if (this.zones[id].lock_button_unlock == button) this.setLock(id,this.DRV_LOCK_NO);
                    if (this.zones[id].lock_button_temporary == button) this.setLock(id,this.DRV_LOCK_TILL_SETPOINTCHANGE);
                }
            }
        }
    
        /* Save zone status on shared */
        this.writeOnShared = function (_id) {

            var toSave = [];

            for (var id in this.zones) {

                var sph;    // set point with hysteresis

                switch (this.zones[id].mode) {

                    case mhOWN.OWN_THERMO_HEATING:
                        sph = "(-" + (this.heat_hysteresis).toString() + ")";
                        break;

                    case mhOWN.OWN_THERMO_COOLING:
                        sph = "(+" + (this.cool_hysteresis).toString() + ")";
                        break;
                }

                var data = {

                    'nam': String(this.zones[id].name),
                    'hid': String(this.zones[id].hvac_address),
                    'mgw': String(this.zones[id].mh_ownserver),
                    'mid': String(this.zones[id].mh_address),
                    'mta': (this.zones[id].mh_ambient > 0) ? "%.1f".sprintf(this.zones[id].mh_ambient) : "n.a.",
                    'hta': (this.hvac.getAmbient(this.zones[id].zoneid) > 0) ? "%.1f".sprintf(this.hvac.getAmbient(this.zones[id].zoneid)) : "n.a.",
                    'sta': (this.zones[id].mh_setpoint > 0) ? "%.1f".sprintf(this.zones[id].mh_setpoint) : "n.a.",
                    'mfs': String(HVAC_Driver.s_fanspeeds[this.zones[id].mh_fanspeed]),
                    'hfs': String(HVAC_Driver.s_fanspeeds[this.hvac.getFanSpeed(this.zones[id].zoneid)]),
                    'mmd': HVAC_Driver.s_modes[this.zones[id].mode],
                    'hmd': HVAC_Driver.s_modes[this.hvac.getOperationMode(this.zones[id].zoneid)],
                    'hpw': (this.hvac.getPower(this.zones[id].zoneid)) ? "ON" : "OFF",
                    'mpw': (this.zones[id].power) ? "ON" : "OFF",
                    'lck': this.zones[id].lock,
                    'his': this.hvac.getHWExist(this.zones[id].zoneid),
                    'err': this.zones[id].alarm,
                    'fil': this.zones[id].filter,
                    'ext': this.zones[id].extra,
                    'hys': this.zones[id].hysteresis,
                    'sph': sph,
                    'tsa': (this.zones[id].bac_area == 0 && this.zones[id].bac_address == 0) ? "- / -" : "%d / %d".sprintf(this.zones[id].bac_area, this.zones[id].bac_address)
                }

                //  this.zones[id].led_status

                toSave.push(data);

                /*
                for( var property in data ) {
                    if( data.hasOwnProperty( property ) ) {
                        logger.info( "%s[] shared -> [%s]: %s".sprintf( this.prefix, property, data[property] ) );
                    }
                }
                */
            }
      
            shared.write( 'zones', JSON.stringify( toSave ) );
            shared.write( 'gateway', JSON.stringify( {'connected': this.hvac.isConnected()} ) );
        }

        // +Better Er2 management
        this.startupLedStatusUpdateTimer = new mhTimer();
        this.startupLedStatusUpdateTimer.setInterval(5000, true); // single shot timer
        this.startupLedStatusUpdateTimer.setAction(this.startupLedStatusUpdater.bind(this));
        // -Better Er2 management

    };
  
    return drv;    
})();

HVAC_Driver.OWN_THERMO_DRY = 101;
HVAC_Driver.OWN_THERMO_FAN = 102;
HVAC_Driver.OWN_THERMO_AUTO = 103;          // Normal Auto  

HVAC_Driver.s_modes = [];
HVAC_Driver.s_modes[0] = "OFF"; 
HVAC_Driver.s_modes[1] = "HEAT"; 
HVAC_Driver.s_modes[2] = "COOL"; 
HVAC_Driver.s_modes[101] = "DRY"; 
HVAC_Driver.s_modes[102] = "FAN"; 
HVAC_Driver.s_modes[103] = "AUTO";

HVAC_Driver.OWN_THERMO_INTEGRATE = "INTEGRATE";
HVAC_Driver.OWN_THERMO_SINGLE = "SINGLE";
HVAC_Driver.OWN_THERMO_FANONLY = "FANONLY";
HVAC_Driver.OWN_THERMO_NONE = "OFF";

HVAC_Driver.REACHED_OFF = 100;
HVAC_Driver.REACHED_FAN_LOW = mhOWN.OWN_THERMO_FANSPEED_LOW;
HVAC_Driver.REACHED_FAN_AUTO = mhOWN.OWN_THERMO_FANSPEED_AUTO;
HVAC_Driver.REACHED_NONE = 99;
HVAC_Driver.REACHED_MODE_AUTO = 98;

HVAC_Driver.s_fanspeeds = [];
HVAC_Driver.s_fanspeeds[0]="AUTO"; 
HVAC_Driver.s_fanspeeds[1]="LOW"; 
HVAC_Driver.s_fanspeeds[2]="MED"; 
HVAC_Driver.s_fanspeeds[3] = "HIGH";

HVAC_Driver.s_locks = []; 
HVAC_Driver.s_locks[1] = "LOCK"; 
HVAC_Driver.s_locks[0] = "NO LOCK"; 
HVAC_Driver.s_locks[2] = "LOCK TILL CHANGE";

HVAC_Driver.REACHED_VALUES_DESCRIPTION = [];
HVAC_Driver.REACHED_VALUES_DESCRIPTION[0] = "Turn OFF the unit";
HVAC_Driver.REACHED_VALUES_DESCRIPTION[1] = "Set AUTO fan speed";
HVAC_Driver.REACHED_VALUES_DESCRIPTION[2] = "Set LOW fan speed";
HVAC_Driver.REACHED_VALUES_DESCRIPTION[3] = "No action";
HVAC_Driver.REACHED_VALUES_DESCRIPTION[4] = "Set AUTO mode";

HVAC_Driver.REACHED_VALUES_INDEXES = [];
HVAC_Driver.REACHED_VALUES_INDEXES[0] = 100;
HVAC_Driver.REACHED_VALUES_INDEXES[1] = 0;
HVAC_Driver.REACHED_VALUES_INDEXES[2] = 1;
HVAC_Driver.REACHED_VALUES_INDEXES[3] = 99;
HVAC_Driver.REACHED_VALUES_INDEXES[4] = 98;

HVAC_Driver.lg_modes = [];
HVAC_Driver.lg_modes[0] = "unused";
HVAC_Driver.lg_modes[1] = "COOL";
HVAC_Driver.lg_modes[2] = "DRY";
HVAC_Driver.lg_modes[3] = "FAN";
HVAC_Driver.lg_modes[4] = "AUTO";
HVAC_Driver.lg_modes[5] = "HEAT";

HVAC_Driver.modesDescription = [];
HVAC_Driver.modesDescription[0] = "OFF";
HVAC_Driver.modesDescription[1] = "HEATING";
HVAC_Driver.modesDescription[2] = "COOLING";
HVAC_Driver.modesDescription[10] = "MANUAL";
HVAC_Driver.modesDescription[11] = "AUTO";
HVAC_Driver.modesDescription[12] = "COMFORT";
HVAC_Driver.modesDescription[13] = "ECO";
HVAC_Driver.modesDescription[14] = "PROTECTION";
HVAC_Driver.modesDescription[101] = "DRY";
HVAC_Driver.modesDescription[102] = "FAN";
HVAC_Driver.modesDescription[103] = "AUTO";